package io.aalam.common;

import java.util.logging.*;
import java.util.Set;
import java.util.HashSet;

import org.json.JSONArray;

/**
This class defines the permissions for urls and will an object of this class
will be used for every URL registration.
*/
public class Permissions {
    String condition;
    String[] permissions;
    boolean deny_anon;
    Set<String> deny_exc;
    static final Logger log = Logger.getLogger("java-common");

    /**
    Intialize an empty permissions object
    */
    public Permissions() {
        this.condition = "";
        this.permissions = new String[]{};
        this.deny_anon = false;
        this.deny_exc = new HashSet<String>();
    }

    /**
    Use the static methods all() or any() instead of this method
    */
    private Permissions(String condition, String[] permissions) {
        this.condition = condition;
        this.permissions = permissions;
        this.deny_anon = false;
        this.deny_exc = new HashSet<String>();
    }

    /**
    Initialize a new permissions object with a condition 'all'.
    @param args List of permissions in the format
                "permission-group-name/permission-name"
    */
    public static Permissions all(String... args) {
        String[] perms = new String[args.length];
        int index = 0;
        for (String a : args) {
            perms[index++] = a;
        }
        return new Permissions("all", perms);
    }

    /**
    Initialize a new permissions object with a condition 'all'.
    @param args List of permissions in the format
                "permission-group-name/permission-name"
    */
    public static Permissions any(String... args) {
        String[] perms = new String[args.length];
        int index = 0;

        for (String a : args) {
            perms[index++] = a;
        }
        return new Permissions("any", perms);
    }

    /**
    Deny anonymous requests on a URL

    @return Returns 'this' object so that you can chain like
            Permissions.any("pg1/p1", "pg2/p2").deny_anon()
    */
    public Permissions deny_anon() {
        this.deny_anon = true;
        return this;
    }

    /**
    Deny request from any source except the apps mentioned in the args

    @param args List of apps in the format 'provider-code/app-code'
    @return Returns 'this' object so that you can chain like
            Permissions.any("pg1/p1", "pg2/p2").deny_anon().deny_exc('aalam/base')
    */
    public Permissions deny_exc(String... args) {
        for (String a : args) {
            this.deny_exc.add(a);
        }
        return this;
    }

    /**
    Returns the fully qualified permission name. Fully qualified permission
    name will identify a permission to which app and which permission group in
    the app that this permission belongs to.
    */
    public static String framePermission(String provider, String app,
                                         String permissionGroup,
                                         String permission) {
        return provider + "/" +
               app + "/" +
               permissionGroup + "/" +
               permission;
    }

    /**
    Breaks the fully qualified permission name.

    @param permission - Fully qualiified permission name
    @return list of elements in the permission name
    */
    public static String[] breakPermission(String permission) {
        return permission.split("/");
    }

    /**
    This method is used internall in the framework and the application has
    no use for this method.
    */
    public boolean check(HttpRequest request) {
        /* Check if the user has the necessary permissions to access this url */
        if (this.deny_anon && !request.auth().isAnonymous()) {
            return false;
        }

        if (!this.deny_exc.isEmpty()) {
            if (!this.deny_exc.contains(request.auth().from())) {
                return false;
            }
        }

        for (int index = 0; index < this.permissions.length; index++) {
            String[] p = permissions[index].split("/");
            Integer permId = RoleCache.get(Permissions.framePermission(
                Config.appProviderCode, Config.appCode, p[0], p[1]));

            if (permId == null) {
                log.severe(
                    "Cannot find permission '" +
                    this.permissions[index] +
                    "'");
                return false;
            }

            if (this.condition == "all" && !request.userPermissions().contains(permId)) {
                log.severe("User does not enough permission to access this url");
                return false;
            } else if (this.condition == "any" && request.userPermissions().contains(permId)) {
                break;
            }
        }

        return true;
    }
}
