package io.aalam.common;

import io.netty.bootstrap.ServerBootstrap;
import io.netty.channel.Channel;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.MultithreadEventLoopGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollServerDomainSocketChannel;
import io.netty.channel.unix.DomainSocketAddress;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.handler.logging.LoggingHandler;
import io.netty.handler.logging.LogLevel;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.Files;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.FileSystems;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.GroupPrincipal;
import java.nio.file.LinkOption;
import java.nio.file.attribute.UserPrincipalLookupService;
import java.io.File;
import java.util.Set;
import java.util.HashSet;
import java.net.InetSocketAddress;
import java.net.SocketAddress;


public final class HttpServer {
    static EventLoopGroup bossGroup;
    static EventLoopGroup workerGroup;
    static Channel channel;
    public static final String WSGI_SOCKET_ROOT = "/tmp";

    public HttpServer() {
    }

    private static SocketAddress getSocketAddress() throws Exception {
        Config.newSection(Config.appCode);
        Integer port = Integer.parseInt(Config.getArg(Config.appCode, "port", "-1"));
        if (port > 0) {
            bossGroup = new NioEventLoopGroup(1);
            workerGroup = new NioEventLoopGroup();
            return new InetSocketAddress(port);
        }

        /* We need to bind to a unix socket */
        Path socketPath = Paths.get(WSGI_SOCKET_ROOT).
            resolve(Config.appProviderCode).
            resolve(Config.appCode +
                    ((Launcher.state == Callbacks.STATE_POST_MIGRATE)?
                      "_post_migrate.sock":
                      ".sock"));
        Files.deleteIfExists(socketPath);
        if (Files.notExists(socketPath.getParent())) {
            File dir = socketPath.getParent().toFile();
            dir.mkdir();
            dir.setReadable(true, false);
            dir.setWritable(true, false);
        }

        bossGroup = new EpollEventLoopGroup(1);
        workerGroup = new EpollEventLoopGroup();
        return new DomainSocketAddress(socketPath.toFile());
    }

    public static void run() throws Exception {
        SocketAddress bindParam = getSocketAddress();
        Class channelClass = (bindParam instanceof InetSocketAddress)?
            NioServerSocketChannel.class:
            EpollServerDomainSocketChannel.class;
        try {
            ServerBootstrap b = new ServerBootstrap();
            b.group((EventLoopGroup)bossGroup, (EventLoopGroup)workerGroup)
             .channel(channelClass)
             .handler(new LoggingHandler(LogLevel.INFO))
             .childHandler(new HttpServerInitializer());


            channel = b.bind(bindParam).sync().channel();

            if (bindParam instanceof DomainSocketAddress) {
                Path socketPath = Paths.get(((DomainSocketAddress)bindParam).path());
                UserPrincipalLookupService lookupService =
                     socketPath.getFileSystem().
                     getUserPrincipalLookupService();
                GroupPrincipal group = lookupService.
                    lookupPrincipalByGroupName("www-data");
                Files.getFileAttributeView(
                    socketPath,
                    PosixFileAttributeView.class,
                    LinkOption.NOFOLLOW_LINKS).setGroup(group);
                Set<PosixFilePermission> perm = new HashSet<PosixFilePermission>();
                perm.add(PosixFilePermission.GROUP_READ);
                perm.add(PosixFilePermission.GROUP_WRITE);
                perm.add(PosixFilePermission.GROUP_EXECUTE);
                perm.add(PosixFilePermission.OWNER_READ);
                perm.add(PosixFilePermission.OWNER_WRITE);
                perm.add(PosixFilePermission.OWNER_EXECUTE);
                Files.setPosixFilePermissions(socketPath, perm);
            }

            try {
                channel.closeFuture().sync();
            } catch (java.util.concurrent.CancellationException e) {
                /*Nothing to be done here*/
            }
        } finally {
            bossGroup.shutdownGracefully();
            workerGroup.shutdownGracefully();
            workerGroup.awaitTermination(
                Integer.MAX_VALUE, java.util.concurrent.TimeUnit.SECONDS);
        }
    }

    public static void closeChannel() {
        ((io.netty.channel.DefaultChannelPromise)channel.closeFuture()).cancel(true);
    }
}
