package io.aalam.common;

import io.netty.handler.codec.http.DefaultFullHttpResponse;
import io.netty.handler.codec.http.FullHttpResponse;
import io.netty.handler.codec.http.HttpVersion;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.util.CharsetUtil;

import static io.netty.handler.codec.http.HttpHeaders.Names.*;
import static io.netty.handler.codec.http.HttpResponseStatus.*;
import static io.netty.handler.codec.http.HttpVersion.*;


public class HttpResponse extends DefaultFullHttpResponse {
    private HttpRequest httpRequest;

    /**
    Sets the request object for which this response is being sent
    */
    public void setHttpRequest(HttpRequest req) {
        this.httpRequest = req;
    }

    /**
    Get the request object for this response object.

    @return HttpRequest object
    @see HttpRequest
    */
    public HttpRequest getHttpRequest() {
        return this.httpRequest;
    }

    /**
    Initialize a new response object

    @param req The HttpRequest object
    @param version The HTTP version
    @param status The status code and string for the response
    @param buf The response data

    @see HttpResponseStatus
    */
    public HttpResponse(HttpRequest req, HttpVersion version,
                        HttpResponseStatus status, ByteBuf buf) {
        super(version, status, buf);
        this.httpRequest = req;
        this.headers().set(CONTENT_LENGTH, this.content().readableBytes());
    }

    /**
    Initialize a new response object that has no data

    @param req The HttpRequest object
    @param version The HTTP version
    @param status The status code and string for the response

    @see HttpResponseStatus
    */
    public HttpResponse(HttpRequest req, HttpVersion version,
                        HttpResponseStatus status) {
        this(req, version, status, Unpooled.buffer());
    }

    /**
    Set the content type of the response data

    @param mimeType The mime for the response content
    */
    public void setContentType(String mimeType) {
        this.headers().set(CONTENT_TYPE, mimeType);
    }

    /**
    Set the data be sent in the response.

    @param data The actual data to be sent.
    */
    public void setContent(byte[] data) {
        this.content().writeBytes(Unpooled.copiedBuffer(data));
        this.headers().set(CONTENT_LENGTH, data.length);
    }

    public void setContent(ByteBuf data) {
        this.content().writeBytes(Unpooled.copiedBuffer(data));
        this.headers().set(CONTENT_LENGTH, data.readableBytes());
        this.content().setBytes(0, data);
    }

    /**
    Returns a pre initialized response object with status code "404"
    */
    public static HttpResponse notFound() {
        HttpResponse ret = new HttpResponse(null, HTTP_1_1, NOT_FOUND);
        ret.headers().set(CONTENT_LENGTH, 0);
        return ret;
    }

    /**
    Returns a pre initialized response object with status code "500"
    */
    public static HttpResponse internalServerError() {
        HttpResponse ret = new HttpResponse(null, HTTP_1_1, INTERNAL_SERVER_ERROR);
        ret.headers().set(CONTENT_LENGTH, 0);
        return ret;
    }

    /**
    Returns a pre initialized response object with status code "401"
    */
    public static HttpResponse unAuthorized() {
        HttpResponse ret = new HttpResponse(null, HTTP_1_1, UNAUTHORIZED);
        ret.headers().set(CONTENT_LENGTH, 0);
        return ret;
    }

    /**
    Returns a pre initialized response object with status code "403"
    */
    public static HttpResponse forbidden() {
        HttpResponse ret = new HttpResponse(null, HTTP_1_1, FORBIDDEN);
        ret.headers().set(CONTENT_LENGTH, 0);
        return ret;
    }

    /**
    Get the status code set on this response object

    @return integer status code
    */
    public int statusCode() {
        return this.status().code();
    }
}
