package io.aalam.common;

import io.netty.buffer.ByteBuf;
import io.netty.handler.codec.http.*;
import io.netty.handler.codec.http.DefaultFullHttpRequest;

import io.aalam.common.router.RouteMatch;

import java.util.Map;
import java.util.List;
import java.lang.String;
import java.util.HashSet;


/**
Class to access the request details. This class inherits from netty's
DefaultFullHttpRequest

@see DefaultFullHttpRequest
*/
public class HttpRequest extends DefaultFullHttpRequest {
    int dbSession;
    String path;
    Map<String, List<String>> params;
    RouteMatch routeMatch;
    HashSet<Integer> userPermissions;
    String staticUrlResource;
    String staticFileName;
    Auth auth;

    private void parsePath() {
        QueryStringDecoder qdec = new QueryStringDecoder(getUri());
        this.path = qdec.path();
        this.params = qdec.parameters();
    }

    /**
    Application code will not have any use with the constructors as the application
    code will always process on an initialized request object.
    */
    public HttpRequest(HttpVersion httpVersion, HttpMethod method, String uri, ByteBuf content) {
        super(httpVersion, method, uri, content);
        this.parsePath();
        this.auth = null;
    }

    public HttpRequest(FullHttpRequest req) {
        super(req.getProtocolVersion(), req.getMethod(), req.getUri(), req.content().duplicate());
        this.headers().set(req.headers());
        this.trailingHeaders().set(req.trailingHeaders());
        req.content().retain();
        this.parsePath();
    }

    /**
    Returns the URL path, example, for url
    GET /aalam/base/users?param1=value&amp;param2=value, the path will be
    /aalam/base/users

    @return String value of the path
    */
    public String path() {
        return this.path;
    }

    /**
    Returns the URL parameters.

    @return Map of string to list of strings - with the key being the parameter name and
                                value being the parameter value.
    */
    public Map<String, List<String>> params() {
        return params;
    }

    /**
    Application will have no use with this method. this method will be used
    by the framework internally.
    */
    public void setRouteMatch(RouteMatch obj) {
        this.routeMatch = obj;
    }

    /**
    Application will have no use with this method. this method will be used
    by the framework internally.
    */
    public RouteMatch routeMatch() {
        return this.routeMatch;
    }

    /**
    Get the HTTP method

    @return String name of the method
    */
    public String methodName() {
        return this.getMethod().name();
    }

    /**
    Application will have no use with this method. this method will be used
    by the framework internally.
    */
    public void setUserPermissions(HashSet<Integer> permissions) {
        this.userPermissions = permissions;
    }

    /**
    Get the permissions of the user initiating this request

    @return Set of permission ids.
    */
    public HashSet<Integer> userPermissions() {
        return this.userPermissions;
    }

    /**
    Though applications can pre-inform the static URL and the way to access the
    static data through build plugin, there might be some situation which would
    want to send a static file on a non-static URL. For example, if an
    application has a static-url prefix as `/aalam/base/s/`.
    Suppose the application wants to serve a static file in a URL
    that does not have a prefix `/aalam/base/s/`, or if the file is not on
    the usual place that the framework can find, it can inform the framework
    about this static resource through this method

    This method needs to be called in the action handler. By doing so, the
    framework will take care of sending the response.

    @param resource A unique name of this resource. This name should not clash
                    with any of the other static resources.
    @param file The path to the resource
    */
    public void setStaticInfo(String resource, String file) {
        this.staticUrlResource = resource;
        this.staticFileName = file;
    }

    /**
    Get the static resource set in this request

    @return The resource name
    */
    public String getStaticUrlResource() {
        return this.staticUrlResource;
    }

    /**
    Get the static file set in this request

    @return The resource's absolute file path
    */
    public String getStaticFile() {
        return this.staticFileName;
    }

    /**
    The authentication parameters for a request.
    */
    public class Auth {
        Integer userId;
        String email;
        String appId;
        String from;
        String authType;

        /**
        Authenication parameters will be initialized by the framework. So, the
        application will never use this constructor
        */
        public Auth(String type) {
            this.authType = type;
        }

        /**
        ID of the user if obtained

        @return integer value of the user ID
        */
        public Integer userId() {
            return userId;
        }

        /**
        Email of the user requesting

        @return string email id of the requesting user
        */
        public String email() {
            return email;
        }

        /**
        If the URL is requested from other apps or an external source,
        this method will identify the remote source. If the request is
        made by internal applications, from will return the app source in the
        format 'provider_code/app_code'

        @return String remote requester's name
        */
        public String from() {
            return from;
        }

        /**
        Check if the request made by other applications

        @return True if made by other applications else False.
        */
        public boolean isInternal() {
            return authType == "internal";
        }

        /**
        Check if the request made from external source

        @return Boolean, true if request by external source else false.
        */
        public boolean isExternalServer() {
            return authType == "external";
        }

        /**
        Check if the request is made by a human user identified by HTTP cookies

        @return Boolean, true if authenticated using cookie else false.
        */
        public boolean isCookie() {
            return authType == "cookie";
        }

        /**
        Check if the request is made by an unauthenticated source.

        @return Boolean, true if request is anonymous else false.
        */
        public boolean isAnonymous() {
            return authType == "anonymous";
        }
    }

    /**
    Returns the auth object of this request.
    */
    public Auth auth() {
        return this.auth;
    }

    /**
    This method will be used by the framework internally.
    */
    public void setCookieAuth(String email, Integer userId) {
        if (this.auth != null)
            return;

        this.auth = new Auth("cookie");
        this.auth.email = email;
        this.auth.userId = userId;
    }

    /**
    This method will be used by the framework internally.
    */
    public void setCookieAuth(String email) {
        if (this.auth != null)
            return;

        this.auth = new Auth("cookie");
        this.auth.email = email;
    }

    /**
    This method will be used by the framework internally.
    */
    public void setTokenAuth(String appId) {
        if (this.auth != null)
            return;

        this.auth = new Auth("token");
        this.auth.appId = appId;
    }

    /**
    This method will be used by the framework internally.
    */
    public void setInternalAuth(String from, String userEmail) {
        if (this.auth != null)
            return;

        this.auth = new Auth("internal");
        this.auth.from = from;
        this.auth.email = userEmail;
    }

    /**
    This method will be used by the framework internally.
    */
    public void setExternalAuth(String from) {
        if (this.auth != null)
            return;

        this.auth = new Auth("external");
        this.auth.from = from;
    }

    /**
    This method will be used by the framework internally.
    */
    public void setExternalAuth(String from, String userEmail, Integer id) {
        if (this.auth != null)
            return;

        this.auth = new Auth("external");
        this.auth.from = from;
        this.auth.email = userEmail;
        this.auth.userId = id;
    }

    /**
    This method will be used by the framework internally.
    */
    public void setAnonymousAuth() {
        if (this.auth != null)
            return;

        this.auth = new Auth("anonymous");
        this.auth.email = "anonymous";
    }
}
