package io.aalam.common;

import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.http.DefaultFullHttpResponse;
import io.netty.handler.codec.http.FullHttpResponse;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.netty.util.CharsetUtil;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;

import static io.netty.handler.codec.http.HttpHeaders.Names.*;
import static io.netty.handler.codec.http.HttpResponseStatus.*;
import static io.netty.handler.codec.http.HttpVersion.*;


public abstract class HttpPreProcessor extends SimpleChannelInboundHandler<HttpRequest> {
    @Override
    public void channelReadComplete(ChannelHandlerContext ctx) {
        ctx.fireChannelReadComplete();
    }

    @Override
    protected void channelRead0(ChannelHandlerContext ctx, HttpRequest msg) throws Exception {
        Object ret = this.process(msg);
        if (ret != null) {
            this.sendResponse(ctx, msg, ret);
        } else {
            ctx.fireChannelRead(msg.retain());
        }
    }

    protected void sendResponse(ChannelHandlerContext ctx, HttpRequest req,
                                HttpResponseStatus status, Object data) {
        if (data instanceof HttpResponse) {
            /* Some handler in the pipeline want to break the chain */
            ctx.write(data);
            return;
        }

        HttpResponse response = new HttpResponse(
            req, HTTP_1_1, status,
            Unpooled.copiedBuffer(data.toString(), CharsetUtil.UTF_8));
        ctx.fireChannelRead(response.retain());
    }

    protected void sendResponse(ChannelHandlerContext ctx, HttpRequest req, Object data) {
        this.sendResponse(ctx, req, OK, data);
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        cause.printStackTrace();
        ctx.close();
    }

    public abstract Object process(HttpRequest msg) throws Exception;
}
