package io.aalam.common;


import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.MultithreadEventLoopGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollDomainSocketChannel;
import io.netty.channel.unix.DomainSocketAddress;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.handler.codec.http.DefaultFullHttpRequest;
import io.netty.handler.codec.http.HttpContent;
import io.netty.handler.codec.http.LastHttpContent;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpHeaderValues;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.HttpRequest;
import io.netty.handler.codec.http.HttpVersion;
import io.netty.handler.codec.http.cookie.ClientCookieEncoder;
import io.netty.handler.codec.http.cookie.DefaultCookie;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.ChannelPipeline;
import io.netty.channel.socket.SocketChannel;
import io.netty.handler.codec.http.HttpClientCodec;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.http.HttpUtil;
import io.netty.handler.codec.http.HttpObject;
import io.netty.handler.codec.http.HttpObjectAggregator;
import io.netty.handler.codec.http.QueryStringEncoder;
import io.netty.buffer.Unpooled;
import io.netty.util.CharsetUtil;

import java.net.URI;
import java.util.Map;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.io.File;


class HttpResponseHolder {
    HttpResponse response;

    public void setResponse(HttpResponse _response) {
        this.response = _response;
    }

    public HttpResponse getResponse() {
        return response;
    }
}


class HttpClientHandler extends SimpleChannelInboundHandler<HttpObject> {
    HttpResponseHolder response;

    public HttpClientHandler(HttpResponseHolder response) {
        this.response = response;
    }

    @Override
    public void channelRead0(ChannelHandlerContext ctx, HttpObject msg) {
        if (msg instanceof io.netty.handler.codec.http.HttpResponse) {
            io.netty.handler.codec.http.HttpResponse resp = (io.netty.handler.codec.http.HttpResponse)msg;
            HttpResponse bmboResponse = new HttpResponse(null, resp.getProtocolVersion(),
                                        resp.status());
            bmboResponse.headers().set(resp.headers());
            this.response.setResponse(bmboResponse);
        }
        if (msg instanceof HttpContent) {
            HttpContent content = (HttpContent) msg;

            this.response.getResponse().setContent(
                content.content());

            if (content instanceof LastHttpContent) {
                ctx.close();
            }
        }
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        cause.printStackTrace();
        ctx.close();
    }
}

class HttpClientInitializer extends ChannelInitializer<SocketChannel> {
    HttpResponseHolder response;

    public HttpClientInitializer(HttpResponseHolder response) {
        this.response = response;
    }

    @Override
    public void initChannel(SocketChannel ch) {
        ChannelPipeline p = ch.pipeline();

        p.addLast(new HttpClientCodec());
        p.addLast(new HttpClientHandler(response));
    }
}

public class HttpClient {
    private static SocketAddress getSocketAddress(String host, int port) {
        if (port > 0) {
            return new InetSocketAddress(host, port);
        }

        /* We need to bind to a unix socket */
        System.loadLibrary("netty-transport-native-epoll");
        String path = host.replaceAll("%2F", "/");
        path = path.replace("%2f", "/");
        return new DomainSocketAddress(new File(path));
    }

    public static HttpResponse request(
            String method, String url, Map<String, String> headers,
            Map<String, String> params, byte[] data) throws Exception {
        URI uri = new URI(url);
        String scheme = uri.getScheme() == null? "http" : uri.getScheme();
        String host = uri.getHost() == null? "127.0.0.1" : uri.getHost();
        String path = uri.getPath();
        HttpResponseHolder responseHolder = new HttpResponseHolder();

        int port = uri.getPort();
        if (port == -1) {
            if ("http".equalsIgnoreCase(scheme)) {
                port = 80;
            }
        }

        if (!"http".equalsIgnoreCase(scheme) && !"http+unix".equalsIgnoreCase(scheme)) {
            System.err.println("Only http/http+unix is supported.");
            return null;
        }

        EventLoopGroup group = (scheme == "http+unix")?
            new EpollEventLoopGroup():
            new NioEventLoopGroup();

        try {
            Bootstrap b = new Bootstrap();
            b.group(group)
             .channel((scheme == "http+unix"?
                EpollDomainSocketChannel.class:
                NioSocketChannel.class))
             .handler(new HttpClientInitializer(responseHolder));

            Channel ch = b.connect(getSocketAddress(host, port)).sync().channel();

            QueryStringEncoder enc = new QueryStringEncoder(path);
            if (params != null) {
                for (Map.Entry<String, String> entry : params.entrySet()) {
                    enc.addParam(entry.getKey(), entry.getValue());
                }
            }

            HttpRequest request;
            if (data != null) {
                request = new DefaultFullHttpRequest(
                    HttpVersion.HTTP_1_1, new HttpMethod(method), enc.toString(),
                    Unpooled.copiedBuffer(data));
            } else {
                request = new DefaultFullHttpRequest(
                    HttpVersion.HTTP_1_1, new HttpMethod(method), enc.toString());
            }
            request.headers().set(HttpHeaderNames.HOST, (scheme=="http+unix")?"localhost":host);
            request.headers().set(HttpHeaderNames.CONNECTION, HttpHeaderValues.CLOSE);

            if (headers != null) {
                for (Map.Entry<String, String> entry : headers.entrySet()) {
                    request.headers().set(entry.getKey(), entry.getValue());
                }
            }

            ch.writeAndFlush(request);

            ch.closeFuture().sync();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            group.shutdownGracefully();
            return responseHolder.getResponse();
        }
    }
}
