package io.aalam.common;

import io.netty.util.CharsetUtil;
import io.netty.channel.ChannelHandler.Sharable;
import io.netty.handler.codec.http.QueryStringDecoder;
import io.netty.buffer.Unpooled;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.HttpResponseStatus;
import static io.netty.handler.codec.http.HttpHeaders.Names.*;
import static io.netty.handler.codec.http.HttpResponseStatus.*;
import static io.netty.handler.codec.http.HttpVersion.*;

import io.aalam.common.serializers.JsonSerializer;
import org.yaml.snakeyaml.Yaml;

import java.util.ArrayList;
import java.util.regex.Pattern;
import java.util.regex.Matcher;
import java.nio.file.Path;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.Set;
import java.util.HashSet;
import java.io.FileReader;
import java.io.File;
import java.util.HashMap;
import java.io.FileInputStream;
import java.io.InputStream;

import org.json.JSONObject;
import org.json.JSONArray;
import org.json.JSONTokener;


abstract class Hooks extends JsonSerializer {
    public String url;
    Pattern pattern;
    public String method;
    public char[] type;

    public Hooks(String url, String method, char[] type) {
        this.url = url;
        this.method = method.toUpperCase();
        this.type = type;
        this.pattern = Pattern.compile(url.replace("*", "[^/]*?"));
    }

    public boolean match(HttpRequest request, char hookType) {
        Matcher m =  this.pattern.matcher(request.path());
        if (!m.matches())
            return false;

        if (!request.methodName().toUpperCase().equals(this.method))
            return false;

        if (hookType == 0)
            return true;

        for(int index = 0; index < this.type.length; index++) {
            if (hookType == this.type[index]) {
                return true;
            }
        }

        return false;
    }

    public byte[] toJson(HttpRequest request, HttpResponse response) {
        JSONObject finalObj = new JSONObject();

        Map<String, String> hdrs = new HashMap<String, String>();
        for (Map.Entry<String, String> entry: request.headers()) {
            hdrs.put(entry.getKey(), entry.getValue());
        }

        if (response != null) {
            finalObj.put("type", "A");
            finalObj.put("data", response.content().toString(CharsetUtil.UTF_8));
            finalObj.put("status", response.status().code());
        } else {
            finalObj.put("type", "B");
            finalObj.put("data", request.content().toString(CharsetUtil.UTF_8));
        }

        finalObj.put("headers", new JSONObject(hdrs));
        finalObj.put("params", new JSONObject(request.params()));
        return finalObj.toString().getBytes();
    }

    public abstract HttpResponse processBefore(HttpRequest request);
    public abstract HttpResponse processAfter(HttpRequest request, HttpResponse response);
}


class RestrictedHooks extends Hooks {
    Set<String> except;

    public RestrictedHooks(String url, String method, char[] type, ArrayList<String> exc) {
        super(url, method, type);
        this.except = new HashSet<String>();
        if (exc != null) {
            for (String s : exc) {
                except.add(s);
            }
        }
    }

    public HttpResponse processBefore(HttpRequest request) {
        return null;
    }

    public HttpResponse processAfter(HttpRequest request, HttpResponse response) {
        return null;
    }

    public boolean isRestricted(UrlHooks h, char type) {
        HttpRequest req = new HttpRequest(
            HTTP_1_1, new HttpMethod(h.method), h.url, Unpooled.buffer());
        if (this.match(req, type)) {
            if (!this.except.contains(h.hooker))
                return true;
        }

        return false;
    }
}


class UrlHooks extends Hooks {
    public String hooker;
    String hookerUrl;

    public UrlHooks(String url, String method, char[] type, String hooker) {
        super(url, method, type);

        this.hooker = hooker;
        this.hookerUrl = "/" + hooker + "/_/hooks";
    }

    public HttpResponse processBefore(HttpRequest request) {
        HashMap<String, String> params = new HashMap<String, String>();
        params.put("method", request.methodName());
        params.put("url", request.path());
        return Utils.requestLocalServer("POST", this.hookerUrl,
                                        params, "application/json",
                                        this.toJson(request, null));
    }

    public HttpResponse processAfter(HttpRequest request, HttpResponse response) {
        HashMap<String, String> params = new HashMap<String, String>();
        params.put("method", request.methodName());
        params.put("url", request.path());
        return Utils.requestLocalServer("POST", this.hookerUrl,
                                        params, "application/json",
                                        this.toJson(request, response));
    }

    public static UrlHooks deserialize(byte[] data) {
        JSONObject jobj = new JSONObject(new String(data));
        return deserialize(jobj);
    }

    public static UrlHooks deserialize(JSONObject jobj) {
        return new UrlHooks(jobj.getString("url"),
                            jobj.getString("method"),
                            jobj.getString("type").toCharArray(),
                            jobj.getString("hooker"));
    }

    public boolean equals(UrlHooks obj) {
        return (obj.method.equals(this.method) &&
                obj.url.equals(this.url) &&
                obj.hooker.equals(this.hooker));
    }
}

class CallbackHooks extends Hooks {
    public String app;
    Method callbackMethod;

    public CallbackHooks(String app, String url, String method, char[] type, String callback) {
        super(url, method, type);

        String[] entry = callback.split(":");
        try {
            Class cls = java.lang.Class.forName(entry[0]);
            this.callbackMethod = cls.getMethod(entry[1], HttpRequest.class);
        } catch(Exception e) {
            /*Nothing to be done here*/
            e.printStackTrace();
        }
    }

    public HttpResponse processBefore(HttpRequest request) {
        HttpResponse response;
        Serializer obj = null;

        try {
            obj = (Serializer)this.callbackMethod.invoke(null, request);
        } catch(Exception e) {
            e.printStackTrace();
        } finally {
            if (obj == null) {
                response = new HttpResponse(request, HTTP_1_1, NO_CONTENT);
            } else {
                response = new HttpResponse(request, HTTP_1_1, OK);
                obj.serialize(request, response);
            }
        }
        return response;
    }

    public HttpResponse processAfter(HttpRequest request, HttpResponse response) {
        /*Nothing to be done here*/
        return null;
    }
}

@Sharable
public class HooksHandler extends HttpPrePostProcessor {
    String prefix;
    ArrayList<CallbackHooks> hooked;
    ArrayList<RestrictedHooks> restricted;
    ArrayList<UrlHooks> hooks;

    private void loadMap() throws Exception {
        if (Config.hooksMap.isEmpty())
            return;

        Path path = FileSystems.getDefault().getPath(Config.hooksMap);
        if (Files.notExists(path))
            return;

        InputStream input = new FileInputStream(new File(
            Config.hooksMap));
        Yaml yaml = new Yaml();
        Map<String, Object> data = (Map<String, Object>)yaml.load(input);
        for (Map.Entry<String, Object> entry : data.entrySet()) {
            if (entry.getKey().equals("hook")) {
                ArrayList<Object> array = (ArrayList<Object>)entry.getValue();
                for (int index = 0; index < array.size(); index++) {
                    Map<String, String> h = (Map<String, String>)array.get(index);
                    CallbackHooks cbh = new CallbackHooks(h.get("app"),
                                                          h.get("url"),
                                                          h.get("method"),
                                                          h.get("type").toCharArray(),
                                                          h.get("handler"));
                    this.hooked.add(cbh);
                }
            } else if (entry.getKey().equals("restrict")) {
                ArrayList<Object> array = (ArrayList<Object>)entry.getValue();
                for (int index = 0; index < array.size(); index++) {
                    Map<String, Object> h = (Map<String, Object>)array.get(index);
                    ArrayList<String> except = h.containsKey("except")?
                        (ArrayList<String>)h.get("except"):
                        null;
                    RestrictedHooks rh = new RestrictedHooks(
                        (String)h.get("url"),
                        (String)h.get("method"),
                        ((String)h.get("type")).toCharArray(),
                        except != null?except:null);
                    this.restricted.add(rh);
                }
            }
        }
    }

    public HttpResponse addHooks(HttpRequest req) {
        JSONArray array = new JSONArray(req.content().toString(CharsetUtil.UTF_8));
        for (int index = 0; index < array.length(); index++) {
            UrlHooks h = UrlHooks.deserialize(array.getJSONObject(index));
            ArrayList<Character> hookTypes = new ArrayList<Character>();
            for (int rindex = 0; rindex < restricted.size(); rindex++) {
                RestrictedHooks r = restricted.get(rindex);
                for (int tindex = 0; tindex < h.type.length; tindex++) {
                    if (r.isRestricted(h, h.type[tindex])) {
                        break;
                    } else {
                        hookTypes.add(new Character(h.type[tindex]));
                    }
                }
            }
            if (hookTypes.isEmpty())
                continue;
            else {
                Character [] Carray = hookTypes.toArray(new Character[hookTypes.size()]);
                char[] carray = new char[hookTypes.size()];
                for (int tindex = 0; tindex < Carray.length; tindex++) {
                    carray[index] = Carray[tindex];
                }
                h.type = carray;
            }

            hooks.add(h);
        }

        return new HttpResponse(req, HTTP_1_1, OK);
    }

    public HttpResponse deleteHooks(HttpRequest req) {
        JSONArray array = new JSONArray(req.content().toString(CharsetUtil.UTF_8));
        for (int index = 0; index < array.length(); index++) {
            int deletable = -1;
            UrlHooks h = UrlHooks.deserialize(array.getJSONObject(index));

            for (int hooksIndex = 0; hooksIndex < hooks.size(); hooksIndex++) {
                if (hooks.get(hooksIndex).equals(h)) {
                    deletable = hooksIndex;
                    break;
                }
            }

            if (deletable >= 0) {
                hooks.remove(deletable);
            }
        }

        return new HttpResponse(req, HTTP_1_1, OK);
    }

    public HttpResponse hookCallback(HttpRequest request) {
        HttpRequest toRequest;
        Map<String, String> map;
        QueryStringDecoder dec = new QueryStringDecoder(request.getUri());
        String httpMeth = dec.parameters().get("method").get(0);
        String httpUrl = dec.parameters().get("url").get(0);

        toRequest = new HttpRequest(HTTP_1_1,
                                    new HttpMethod(httpMeth),
                                    httpUrl,
                                    request.content());

        for (int index = 0; index < hooked.size(); index++) {
            if (hooked.get(index).match(toRequest, (char)0)) {
                return hooked.get(index).processBefore(request);
            }
        }

        return new HttpResponse(request, HTTP_1_1, OK);
    }

    public void processBefore(HttpRequest req) {
        for (int index = 0; index < restricted.size(); index++) {
            if (restricted.get(index).match(req, 'B')) {
                return;
            }
        }

        /*Not restricted, so check in hooks*/
        for (int index = 0; index < hooks.size(); index++) {
            if (hooks.get(index).match(req, 'B')) {
                /*Ignore the response*/
                hooks.get(index).processBefore(req);
            }
        }
    }

    public void processAfter(HttpRequest req, HttpResponse response) {
        for (int index = 0; index < restricted.size(); index++) {
            if (restricted.get(index).match(req, 'A')) {
                return;
            }
        }

        /*Not restricted, so check in hooks*/
        for (int index = 0; index < hooks.size(); index++) {
            if (hooks.get(index).match(req, 'A')) {
                /*Ignore the response*/
                hooks.get(index).processAfter(req, response);
            }
        }
    }

    public HooksHandler() throws Exception {
        this.prefix = "/" + Config.appProviderCode + "/" +
                        Config.appCode + "/_/hooks";
        hooks = new ArrayList<UrlHooks>();
        hooked = new ArrayList<CallbackHooks>();
        restricted = new ArrayList<RestrictedHooks>();
        this.loadMap();
    }

    public Object process(HttpRequest req) {
        System.out.println("HookPreHandler executed");
        if (req.path().equals(prefix)) {
            /*This is a hook url*/
            if (req.methodName().equals("PUT")) {
                return this.addHooks(req);
            } else if (req.methodName().equals("DELETE")) {
                return this.deleteHooks(req);
            } else if (req.methodName().equals("POST")) {
                return this.hookCallback(req);
            } else {
                return HttpResponse.notFound();
            }
        }
        this.processBefore(req);
        return null;
    }

    public Object process(HttpResponse resp) {
        this.processAfter(resp.getHttpRequest(), resp);
        return null;
    }
}
