package io.aalam.common;

import io.aalam.common.router.Mapper;

/**
Every application using this library will have to implement this class
and return an object of that class from it's entry point.
*/
public interface Callbacks {
    /**
    When an application is started in this state, it  means the application
    is started afresh.
    */
    public static final int STATE_STARTED = 1;

    /**
    This state means the application is restarted after recovering from a
    crash. This state is not supported and is reserved for future use.
    */
    public static final int STATE_RESTARTED = 2;

    /**
    This state means the application is getting upgraded and this is the first
    call to the upgradation process. Any initialization needed for the migration
    processs can be done here. In this state the application will not listen on
    any requests from the client.
    */
    public static final int STATE_MIGRATE = 3;

    /**
    This state means the application is getting upgraded and this is the final
    stage of the upgradation process. This state is equivalent to STATE_STARTED
    except that this signifies that the application has just now completed
    migrating its data from the old app.
    */
    public static final int STATE_POST_MIGRATE = 4;

    /**
    This is called mostly from the packager which does the package validation.
    No initialization should be done here. This state expects only the routing
    parameters from the entry-point, so the initialization will fail.

    Right now this state is not used by the packager. This is reserved for
    future use.
    */
    public static final int STATE_VALIDATION = 5;

    /**
    This method is where the URL registration have to be done.
    This callback will be invoked on all application states except
    STATE_MIGRATE.

    @param mapper The mapper object to which the URL should be connected to
    @see          Mapper
    */
    public void routes(Mapper mapper);

    /**
    This callback will invoked when the application is asked to quit.
    When it is invoked, the application will not serve any incoming requests,
    however it will wait till the application finishes serving earlier
    requests. This callback should close up any persistent connections like
    websockets. If the cleanup is proper, the application will quit imediately
    after serving the last HTTP request, else the application will be terminated
    after some time.
    */
    public void cleanup();

    /**
    This callback is invoked as the first step for updating an application.
    Any data migrations from the older version should be done here. While
    migrating, the older application will still be running, so the migration
    should not affect the older application.

    @param fromVersion The older version from which the new version
                       is migrating from
    */
    public void migrate(String fromVersion);

    /**
    This callback is invoked on the new process that is running on state
    STATE_POST_MIGRATE. This is invoked as soon as the old process quits after
    serving all pending connections.
    */
    public void migrateCompleted();
}
