package io.aalam.common;

import io.netty.handler.codec.http.HttpResponseStatus;
import io.netty.handler.codec.http.HttpVersion;
import io.netty.handler.codec.http.cookie.Cookie;
import io.netty.handler.codec.http.cookie.ServerCookieDecoder;
import io.netty.util.CharsetUtil;

import java.util.Arrays;
import java.io.FileReader;
import java.io.File;
import java.io.DataInputStream;
import java.io.InputStreamReader;
import java.io.FileInputStream;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.KeyFactory;
import java.security.interfaces.ECPublicKey;
import java.security.spec.X509EncodedKeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.ECPoint;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.HashMap;

import org.json.JSONArray;
import org.json.JSONObject;

import org.apache.commons.codec.binary.Base64;

import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemReader;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.ECPointUtil;
import org.bouncycastle.jce.spec.ECNamedCurveParameterSpec;
import org.bouncycastle.jce.spec.ECNamedCurveSpec;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1InputStream;

public class AuthHandler extends HttpPreProcessor {
    private String AUTH_COOKIE_NAME = "auth";

    public Object process(HttpRequest req) {
        System.out.println("AuthHandler executed");
        try {
            if (this.handleTokenAuth(req)) {
                return null;
            } else if (this.handleCookieAuth(req)) {
                return null;
            } else if (this.handleInternalAuth(req)) {
                return null;
            } else if (this.handleExternalAuth(req)) {
                return null;
            } else if (this.handleAnonymous(req)) {
                return null;
            }
        } catch (UnAuthorisedException e) {
            e.printStackTrace();
            return new HttpResponse(req, HttpVersion.HTTP_1_1,
                    HttpResponseStatus.UNAUTHORIZED);
        }
        return null;
    }

    public static String getAppEmail(String provider_code, String app_code) {
        return provider_code + "_" + app_code +
               "@" + Config.hostName;
    }

    private boolean handleTokenAuth(HttpRequest req) throws UnAuthorisedException {
        String token = (String) req.headers().get("X-Auth-Token");
        if (token == null) {
            return false;
        }
        String url = "/aalam/base/tokenauth/app/**/token/" + token;
        HttpResponse response = Utils.requestLocalServer("GET", url, null);
        String content = response.content().toString(CharsetUtil.UTF_8);
        JSONObject jsonObj = new JSONObject(content);
        String expiry = (String) jsonObj.get("expiry");
        if (expiry == null) {
            throw new UnAuthorisedException();
        }
        Date expiry_time = new Date(Long.parseLong(expiry));
        Date current_time = new Date();
        if (expiry_time.after(current_time)) {
            throw new UnAuthorisedException();
        }
        req.setTokenAuth((String) ((JSONObject) jsonObj.get("app")).get("id"));
        return true;
    }

    private static PublicKey getECDSAPublicKey(String keyPath) throws Exception {
        PemReader pemReader = new PemReader(new InputStreamReader(new FileInputStream(keyPath)));
        byte[] pem = pemReader.readPemObject().getContent();
        ASN1InputStream input = new ASN1InputStream(pem);
        ASN1Primitive p = input.readObject();
        ASN1Sequence asn1 = ASN1Sequence.getInstance(p);
        byte[] content = asn1.getObjectAt(1).toASN1Primitive().getEncoded();

        content = Arrays.copyOfRange(content, 3, content.length);
        ECNamedCurveParameterSpec spec = ECNamedCurveTable.getParameterSpec("prime192v1");
        KeyFactory kf = KeyFactory.getInstance("ECDSA", "BC");
        ECNamedCurveSpec params = new ECNamedCurveSpec("prime192v1", spec.getCurve(), spec.getG(), spec.getN());
        ECPoint point =  ECPointUtil.decodePoint(params.getCurve(), content);
        ECPublicKeySpec pubKeySpec = new ECPublicKeySpec(point, params);
        ECPublicKey pk = (ECPublicKey) kf.generatePublic(pubKeySpec);
        return pk;
    }

    private static PublicKey getPublicKey(String keyPath) throws Exception {
        return getPublicKey(keyPath, "RSA");
    }

    private static PublicKey getPublicKey(String keyPath, String algorithm) throws Exception {
        PemReader pemReader = new PemReader(new InputStreamReader(new FileInputStream(keyPath)));
        byte[] content = pemReader.readPemObject().getContent();
        X509EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(content);
        KeyFactory kf = KeyFactory.getInstance(algorithm, "BC");
        return kf.generatePublic(publicKeySpec);
    }

    private boolean handleCookieAuth(HttpRequest req) throws UnAuthorisedException {
        String cookiesString = (String) req.headers().get("COOKIE");
        if (cookiesString == null)
            return false;

        /* Remove the COOKIE header, so that the client will not know about it */
        req.headers().remove("COOKIE");

        Set<Cookie> cookies = ServerCookieDecoder.LAX
                .decode(cookiesString);
        Cookie authCookie = null;
        for (Cookie c : cookies) {
            if (AUTH_COOKIE_NAME.equals(c.name())) {
                authCookie = c;
                break;
            }
        }
        if (authCookie == null) {
            return false;
        }

        String[] cookie_params = authCookie.value().split("#");
        String email_id = cookie_params[0];
        String ts = cookie_params[1];
        String signature = cookie_params[2];
        String userkeys_path = Config.userKeysPath + "/" + email_id + ".pub";
        if (!new File(userkeys_path).exists()) {
            /* Ask the base server to cache this detail */
            Map<String, String> params = new HashMap<String, String>();
            params.put("email_id", email_id);

            HttpResponse response = Utils.requestLocalServer(
                "POST", "/aalam/base/central/userkey",
                params);
            if (response.statusCode() != 200) {
                throw new UnAuthorisedException();
            }
        }

        try {
            PublicKey pk = getECDSAPublicKey(userkeys_path);
            String message = email_id + "#" + ts;
            byte[] b = Base64.decodeBase64(signature);
            Signature sig = Signature.getInstance("SHA1withECDSA", "BC");
            sig.initVerify(pk);
            sig.update(message.getBytes());
            if (!sig.verify(b)) {
                throw new UnAuthorisedException();
            }

            req.setCookieAuth(email_id);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (java.security.NoSuchProviderException e) {
            e.printStackTrace();
        } catch (InvalidKeyException e) {
            e.printStackTrace();
        } catch (SignatureException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }

        return false;
    }

    private boolean handleInternalAuth(HttpRequest req) throws UnAuthorisedException {
        String internal = (String) req.headers().get("X-Auth-Internal");
        if (internal == null) {
            return false;
        }
        String[] s = internal.split(";");
        String prefix = s[0];
        String signature = s[1];
        String[] p = prefix.split("/");
        File f = new File(Config.pubKey);
        File parent = f.getParentFile();
        File pubkey = new File(parent, p[0] + "_" + p[1] + ".pub");
        Object ret = verifySignature(pubkey.getAbsolutePath(), internal, req);
        if (ret == null) {
            throw new UnAuthorisedException();
        }
        if (p.length > 2) {
            String email = p[2];
            req.setInternalAuth(prefix, email);
        } else {
            String email = this.getAppEmail(p[0], p[1]);
            req.setInternalAuth(prefix, email);
        }
        return true;
    }

    private Integer _getUserDetails(String email) throws UnAuthorisedException {
        String url = "/aalam/base/users";
        Map<String, String> params = new HashMap<String, String>();
        params.put("fields", "id");
        params.put("email", email);
        HttpResponse response = Utils.requestLocalServer("GET", url, params);
        if (response.getStatus().code() != 200) {
            System.out.println("Unable to get user detailed, base app returned " + response.statusCode());
            throw new UnAuthorisedException();
        }
        String content = response.content().toString(CharsetUtil.UTF_8);
        JSONArray jsonArray = new JSONArray(content);
        return (Integer) jsonArray.get(0);
    }

    private boolean handleExternalAuth(HttpRequest req) throws UnAuthorisedException {
        String signature = (String) req.headers().get("X-Auth-Signature");
        if (signature == null) {
            return false;
        }
        String prefix = signature.substring(0, signature.indexOf(";"));
        boolean ret = false;
        if (prefix.equals("CENTRALPORTAL")) {
            ret = verifySignature(Config.centralPubKey,
                    signature, req);
        } else if (prefix.startsWith("APPSPORTAL")) {
            ret = verifySignature(Config.appsServerPubKey,
                    signature, req);
        }
        if (!ret) {
            throw new UnAuthorisedException();
        }
        if (prefix.contains("/")) {
            String[] s = prefix.split("/");
            prefix = s[0];
            String email = s[1];
            Integer userId = this._getUserDetails(email);
            req.setExternalAuth(prefix, email, userId);
        } else
            req.setExternalAuth(prefix);

        return true;
    }

    private boolean handleAnonymous(HttpRequest req) {
        req.setAnonymousAuth();
        return true;
    }

    private boolean verifySignature(String keyFile, String header, HttpRequest req) {
        String[] s = header.split(";");
        String prefix = s[0];
        String signature = s[1];
        String url;
        String params = getParamStr(req.params());
        if (params.isEmpty()) {
            url = req.path();
        } else {
            url = req.path() + "?" + params;
        }
        String message = prefix + "#" + url;
        PublicKey pk;
        try {
            pk = getPublicKey(keyFile);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }

        MessageDigest md;
        try {
            md = MessageDigest.getInstance("SHA-1");
            md.update(message.getBytes());
            byte[] messageHash = md.digest();
            byte[] b = Base64.decodeBase64(signature);
            Signature sig = Signature.getInstance("NONEwithRSAandMGF1", "BC");
            sig.initVerify(pk);
            sig.update(messageHash);
            return sig.verify(b);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (java.security.NoSuchProviderException e) {
            e.printStackTrace();
        } catch (InvalidKeyException e) {
            e.printStackTrace();
        } catch (SignatureException e) {
            e.printStackTrace();
        }
        return false;
    }

    private String getParamStr(Map<String, List<String>> params) {
        StringBuilder sb = new StringBuilder();
        String dlim = "";
        for (Entry<String, List<String>> e : params.entrySet()) {
            sb.append(dlim);
            sb.append(e.getKey() + "=" + e.getValue().get(0));

            dlim = "&";
        }
        return sb.toString();
    }

    class UnAuthorisedException extends Exception {

    }
}
