/*
 * Decompiled with CFR 0.152.
 */
package driveline;

import driveline.Consumer;
import driveline.DrivelineException;
import driveline.ListConsumer;
import driveline.LoadConsumer;
import driveline.Query;
import driveline.Record;
import driveline.RecordHandler;
import driveline.Stream;
import driveline.StreamId;
import driveline.SyncConsumer;
import driveline.cbor.encoder.CborEncoder;
import driveline.protocol.AppendOptions;
import driveline.protocol.CancelOptions;
import driveline.protocol.ListOptions;
import driveline.protocol.LoadOptions;
import driveline.protocol.QueryOptions;
import driveline.protocol.ServerMessage;
import driveline.protocol.StoreOptions;
import driveline.transport.SyncTransport;
import driveline.transport.Transport;
import driveline.transport.TransportConfig;
import driveline.transport.TransportDelegate;
import driveline.transport.TransportException;
import java.net.URI;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DrivelineClient
implements TransportDelegate {
    private static final Logger log = LoggerFactory.getLogger(DrivelineClient.class);
    private static final int MAX_ALIASES = 256;
    private static final String ERR_COMMAND_FAILED = "cannot send command";
    private static final String SYNC = "syn";
    private static final String CONTINUOUS_QUERY = "sq";
    private static final String QUICK_QUERY = "qq";
    private static final String CANCEL = "can";
    private static final String DEFINE = "def";
    private static final String LIST_KV = "lst";
    private static final String STORE_KV = "st";
    private static final String LOAD_KV = "ld";
    private static final String REMOVE_KV = "rm";
    private static final String REMOVE_MATCHES_KV = "rmk";
    private static final String APPEND_STREAM = "app";
    private static final String LIST_STREAM = "sls";
    private static final String TRUNCATE_STREAM = "trc";
    private final URI endpoint;
    private final Transport transport;
    final AtomicInteger nextConsumerId = new AtomicInteger();
    final StreamId.Factory streamIdFactory = new StreamId.Factory(256);
    private final Map<Long, Consumer> consumers = new HashMap<Long, Consumer>();
    private final ServerMessage.ServerMessageDecoder decoder = new ServerMessage.ServerMessageDecoder();

    private DrivelineClient(URI endpoint, Transport transport) {
        this.endpoint = endpoint;
        this.transport = transport;
    }

    public void start() throws DrivelineException {
        this.start(TransportConfig.getDefault());
    }

    public void start(TransportConfig config) throws DrivelineException {
        try {
            this.transport.connect(this.endpoint, config, this);
        }
        catch (TransportException e) {
            throw new DrivelineException("cannot start client", e);
        }
    }

    public void stop() throws DrivelineException {
        try {
            this.transport.disconnect();
        }
        catch (TransportException e) {
            throw new DrivelineException("cannot stop client", e);
        }
    }

    public Query continuousQuery(String dql, RecordHandler handler) throws DrivelineException {
        return this.continuousQuery(dql, new QueryOptions(), handler);
    }

    public Query continuousQuery(String dql, QueryOptions options, RecordHandler handler) throws DrivelineException {
        return this.query(new Query(this, this.nextConsumerID(), dql, true, options, handler));
    }

    public Query query(String dql, RecordHandler handler) throws DrivelineException {
        return this.query(dql, new QueryOptions(), handler);
    }

    public Query query(String dql, QueryOptions options, RecordHandler handler) throws DrivelineException {
        return this.query(new Query(this, this.nextConsumerID(), dql, false, options, handler));
    }

    Query query(Query query) throws DrivelineException {
        this.registerConsumer(query);
        String command = query.isContinuous ? CONTINUOUS_QUERY : QUICK_QUERY;
        this.sendCommand(CborEncoder.arrayEncoder().encode(command).encode(query.consumerId).encode(query.options).encode(query.query).getBytes());
        return (Query)query.getResult();
    }

    public void cancel(Query query) throws DrivelineException {
        this.cancel(query, new CancelOptions());
    }

    public void cancel(Query query, CancelOptions options) throws DrivelineException {
        Objects.requireNonNull(query);
        Objects.requireNonNull(options);
        if (!this.consumers.containsKey(query.consumerId)) {
            return;
        }
        try {
            this.sendCommand(CborEncoder.arrayEncoder().encode(CANCEL).encode(query.consumerId).encode(options).getBytes());
        }
        finally {
            this.unregisterConsumer(query);
        }
    }

    public Stream openStream(String stream) throws DrivelineException {
        Objects.requireNonNull(stream);
        StreamId streamId = this.streamIdFactory.get(stream);
        if (streamId.isAlias()) {
            this.define(streamId, stream);
        }
        return new Stream(this, streamId);
    }

    public void closeStream(Stream stream) {
        this.streamIdFactory.release(stream.streamId);
    }

    public void append(String stream, byte[] record) throws DrivelineException {
        this.append(StreamId.of(stream), record, new AppendOptions());
    }

    public void append(String stream, byte[] record, AppendOptions options) throws DrivelineException {
        this.append(StreamId.of(stream), record, options);
    }

    public void append(String stream, Collection<byte[]> records) throws DrivelineException {
        this.append(stream, records, new AppendOptions());
    }

    public void append(String stream, Collection<byte[]> records, AppendOptions options) throws DrivelineException {
        StreamId streamId = StreamId.of(stream);
        for (byte[] record : records) {
            this.append(streamId, record, options);
        }
    }

    void append(StreamId streamId, byte[] record, AppendOptions options) throws DrivelineException {
        this.sendCommand(CborEncoder.arrayEncoder().encode(APPEND_STREAM).encode(streamId).encode(options).encode(record).getBytes());
    }

    public void truncate(String stream) throws DrivelineException {
        this.truncate(StreamId.of(stream));
    }

    void truncate(StreamId streamId) throws DrivelineException {
        this.sendCommand(CborEncoder.arrayEncoder().encode(TRUNCATE_STREAM).encodeNull().encode(streamId).getBytes());
    }

    public Iterable<String> listStreams(String streamPattern) throws DrivelineException {
        return this.list(LIST_STREAM, streamPattern, new ListOptions());
    }

    public Iterable<String> listStreams(String streamPattern, ListOptions options) throws DrivelineException {
        return this.list(LIST_STREAM, streamPattern, options);
    }

    public void store(String keyName, byte[] record) throws DrivelineException {
        this.store(keyName, record, new StoreOptions());
    }

    public void store(String keyName, byte[] record, StoreOptions options) throws DrivelineException {
        this.sendCommand(CborEncoder.arrayEncoder().encode(STORE_KV).encode(keyName).encode(options).encode(record).getBytes());
    }

    public CompletableFuture<Record> load(String key) throws DrivelineException {
        return this.load(key, new LoadOptions());
    }

    public CompletableFuture<Record> load(String key, LoadOptions options) throws DrivelineException {
        return this.load(new LoadConsumer(this, this.nextConsumerID(), key, options));
    }

    CompletableFuture<Record> load(LoadConsumer consumer) throws DrivelineException {
        this.registerConsumer(consumer);
        this.sendCommand(CborEncoder.arrayEncoder().encode(LOAD_KV).encode(consumer.consumerId).encode(consumer.getOptions()).encode(consumer.keyName).getBytes());
        return (CompletableFuture)consumer.getResult();
    }

    public void remove(String key) throws DrivelineException {
        this.sendCommand(CborEncoder.arrayEncoder().encode(REMOVE_KV).encodeUndefined().encode(key).getBytes());
    }

    public void removeMatches(String keyPattern) throws DrivelineException {
        this.sendCommand(CborEncoder.arrayEncoder().encode(REMOVE_MATCHES_KV).encodeUndefined().encode(keyPattern).getBytes());
    }

    public Iterable<String> listKeys(String keyPattern) throws DrivelineException {
        return this.list(LIST_KV, keyPattern, new ListOptions());
    }

    public Iterable<String> listKeys(String keyPattern, ListOptions options) throws DrivelineException {
        return this.list(LIST_KV, keyPattern, options);
    }

    private Iterable<String> list(String command, String pattern, ListOptions options) throws DrivelineException {
        ListConsumer consumer = new ListConsumer(this, this.nextConsumerID(), pattern, options);
        this.registerConsumer(consumer);
        this.sendCommand(CborEncoder.arrayEncoder().encode(command).encode(consumer.consumerId).encode(consumer.getOptions()).encode(consumer.pattern).getBytes());
        return (Iterable)consumer.getResult();
    }

    public CompletableFuture<Void> sync() throws DrivelineException {
        SyncConsumer consumer = new SyncConsumer(this, this.nextConsumerID());
        this.registerConsumer(consumer);
        this.sendCommand(CborEncoder.arrayEncoder().encode(SYNC).encode(consumer.consumerId).getBytes());
        return (CompletableFuture)consumer.getResult();
    }

    void define(StreamId streamId, String stream) throws DrivelineException {
        this.sendCommand(CborEncoder.arrayEncoder().encode(DEFINE).encode(streamId).encode(stream).getBytes());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void registerConsumer(Consumer c) {
        Map<Long, Consumer> map = this.consumers;
        synchronized (map) {
            this.consumers.put(c.consumerId, c);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void unregisterConsumer(Consumer consumer) {
        Map<Long, Consumer> map = this.consumers;
        synchronized (map) {
            this.consumers.remove(consumer.consumerId);
        }
    }

    private void sendCommand(byte[] command) throws DrivelineException {
        try {
            this.transport.send(command);
        }
        catch (TransportException e) {
            throw new DrivelineException(ERR_COMMAND_FAILED, e);
        }
    }

    @Override
    public void onConnect() {
    }

    @Override
    public void onReconnect() {
        try {
            Map<Integer, String> aliases = this.streamIdFactory.getAliases();
            for (Map.Entry<Integer, String> entry : aliases.entrySet()) {
                this.define(StreamId.of(entry.getKey()), entry.getValue());
            }
            for (Consumer consumer : this.consumers.values()) {
                Consumer result;
                try {
                    result = consumer.onReconnect();
                }
                catch (DrivelineException ignored) {
                    result = null;
                }
                this.handleConsumerResult(consumer, result);
            }
        }
        catch (Exception e) {
            log.error("failure while reconnecting", (Throwable)e);
        }
    }

    @Override
    public void onDisconnect() {
        for (Consumer consumer : this.consumers.values()) {
            Consumer result;
            try {
                result = consumer.onDisconnect();
            }
            catch (DrivelineException ignored) {
                result = null;
            }
            this.handleConsumerResult(consumer, result);
        }
    }

    @Override
    public void onError(String error) {
        log.error("connection failed: {}", (Object)error);
    }

    @Override
    public void onMessage(byte[] message, int offset, int length) {
        try {
            ServerMessage serverMessage = ServerMessage.fromBytes(this.decoder, message, offset, length);
            Consumer consumer = this.consumers.get(serverMessage.consumerID);
            if (consumer == null) {
                log.error("received data for an unregistered consumer {}", (Object)serverMessage.consumerID);
                return;
            }
            Consumer result = consumer.onMessage(serverMessage);
            this.handleConsumerResult(consumer, result);
        }
        catch (Exception e) {
            log.error("error while ", (Throwable)e);
        }
    }

    private void handleConsumerResult(Consumer current, Consumer next) {
        if (next == null) {
            try {
                current.close();
            }
            catch (DrivelineException drivelineException) {
                // empty catch block
            }
            this.unregisterConsumer(current);
        } else if (current != next) {
            this.registerConsumer(next);
        }
    }

    private int nextConsumerID() {
        return this.nextConsumerId.getAndIncrement();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private URI endpoint;
        private Transport transport;

        public Builder endpoint(String endpoint) {
            return this.endpoint(URI.create(endpoint));
        }

        public Builder endpoint(URI endpoint) {
            if (endpoint == null) {
                throw new IllegalArgumentException("endpoint must be specified");
            }
            try {
                String scheme = endpoint.getScheme();
                if (!scheme.equals("ws") && !scheme.equals("wss")) {
                    throw new IllegalArgumentException("endpoint must be a WebSocket URI");
                }
            }
            catch (Exception e) {
                throw new IllegalArgumentException("invalid endpoint");
            }
            this.endpoint = endpoint;
            return this;
        }

        public Builder transport(Transport transport) {
            if (transport == null) {
                throw new IllegalArgumentException("transport must be specified");
            }
            this.transport = transport;
            return this;
        }

        public DrivelineClient build() {
            Transport t = this.transport != null ? this.transport : new SyncTransport();
            return new DrivelineClient(this.endpoint, t);
        }
    }
}

