/*
 * Copyright 2019, 1533 Systems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package driveline.transport;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.net.URI;
import java.util.LinkedList;
import java.util.Properties;
import java.util.Queue;
import java.util.concurrent.Semaphore;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SyncTransport implements Transport, Runnable {
  private static final Logger log = LoggerFactory.getLogger(SyncTransport.class);
  private URI endpoint;
  private TransportConfig config;
  private TransportDelegate delegate;
  private Socket connection;
  private OutputStream sos;
  private InputStream sis;
  private final Object writeLock = new Object();
  private Queue<byte[]> outputBuffer = new LinkedList<>();
  private Semaphore writeAvailable = new Semaphore(100);
  private Semaphore writerRunnable = new Semaphore(0);

  @Override
  public void connect(URI endpoint, TransportConfig config, TransportDelegate delegate) throws TransportException {
    synchronized (this) {
      if (this.endpoint != null) {
        throw new TransportException("Transport already connected");
      }
      this.endpoint = endpoint;
      this.config = config;
      this.delegate = delegate;
      connect();
      Thread reader = new Thread(this);
      reader.setName("driveline-reader");
      reader.setDaemon(true);
      reader.start();

      startWriter();
    }
  }

  @Override
  public void disconnect() {
    synchronized (this) {
      try {
        if (connection != null)
          connection.close();
      } catch (IOException e) {
        //ignore
      }
      connection = null;
      endpoint = null;
      //todo wait for reader to exit or something
    }
  }

  @Override
  public void send(byte[] data) {
    writeAvailable.acquireUninterruptibly();
    synchronized (writeLock) {
      outputBuffer.add(data);
      writeLock.notify();
    }
  }

  private byte[] sendBuffer = new byte[1024 * 1024 + 65536 + 1024];

  private void sendInternal(Queue<byte[]> list) {
    try {
      int size = 0;
      for (byte[] msg : list) {
        sendBuffer[size++] = (byte) 0x82;
        if (msg.length < 126) {
          sendBuffer[size++] = (byte) msg.length;
          System.arraycopy(msg, 0, sendBuffer, size, msg.length);
          size += msg.length;
        } else if (msg.length < 65536) {
          sendBuffer[size++] = 126;
          sendBuffer[size++] = (byte) (msg.length >>> 8);
          sendBuffer[size++] = (byte) (msg.length);
          System.arraycopy(msg, 0, sendBuffer, size, msg.length);
          size += msg.length;
        } else {
          sendBuffer[size++] = 127;
          sendBuffer[size++] = 0;
          sendBuffer[size++] = 0;
          sendBuffer[size++] = 0;
          sendBuffer[size++] = 0;
          sendBuffer[size++] = (byte) (msg.length >>> 24);
          sendBuffer[size++] = (byte) (msg.length >>> 16);
          sendBuffer[size++] = (byte) (msg.length >>> 8);
          sendBuffer[size++] = (byte) (msg.length);

          synchronized (this) {
            if (sos != null) {
              sos.write(sendBuffer, 0, size);
              sos.write(msg);
            } else return;
          }
          size = 0;
          continue;
        }

        if (size >= 1024 * 1024) {
          synchronized (this) {
            if (sos != null) {
              sos.write(sendBuffer, 0, size);
            } else return;
          }
          size = 0;
        }
      }
      if (size > 0) {
        synchronized (this) {
          if (sos != null) {
            sos.write(sendBuffer, 0, size);
          }
        }
      }
    } catch (IOException ignored) {
    }
  }

  @Override
  public void receive(byte[] data) {
    delegate.onMessage(data, 0, data.length);
  }

  @Override
  public boolean isActive() {
    return connection != null;
  }

  private void startWriter() {
    Thread writer = new Thread(() -> {
      while (true) {
        writerRunnable.acquireUninterruptibly();
        while (sos != null) {
          Queue<byte[]> list;
          try {
            synchronized (writeLock) {
              while (outputBuffer.isEmpty()) {
                writeLock.wait();
              }
              list = outputBuffer;
              outputBuffer = new LinkedList<>();
            }
          } catch (InterruptedException ignored) {
            continue;
          }
          if (list.isEmpty()) {
            continue;
          }
          writeAvailable.release(list.size());
          sendInternal(list);
        }
      }
    });
    writer.setName("driveline-writer");
    writer.setDaemon(true);
    writer.start();
  }

  private void connect() throws TransportException {
    int tries = config.reconnectAttempts;
    while (tries-- > 0) {
      try {
        openConnection();
        InputStream tempSis = connection.getInputStream();
        OutputStream tempSos = connection.getOutputStream();
        tempSos.write(HTTP_HEADER);
        int pos = readFrom(tempSis, HTTP_RESP.length);

        byte[] buf = getReadBuffer();
        for (int i = 0; i < HTTP_RESP.length; ++i) {
          if (buf[pos + i] != HTTP_RESP[i])
            throw new IOException("Invalid response from HTTP endpoint");
        }

        int state = 2; // we saw \r\n
        int limit = 65536;
        while (state < 4 && limit-- > 0) {
          pos = readFrom(tempSis, 1);
          int c = getReadBuffer()[pos];
          if (c == '\r' && (state & 1) == 0) {
            ++state;
          } else if (c == '\n' && (state & 1) == 1) {
            ++state;
          } else {
            state = 0;
          }
        }
        if (state < 4) {
          throw new IOException("Invalid response from HTTP endpoint");
        }

        sos = tempSos;
        sis = tempSis;
        writerRunnable.release();
        return;
      } catch (IOException ignored) {
        closeConnection();
        try {
          Thread.sleep(config.connectionTimeout.toMillis());
        } catch (InterruptedException e) {
          //ignore
        }
      }
    }
    throw new TransportException("Unable to connect to remote host");
  }

  private void openConnection() throws IOException {
    int port = endpoint.getPort();
    if (port == -1) {
      port = endpoint.getScheme().equals("ws") ? 80 : 443;
    }
    connection = new Socket(endpoint.getHost(), port);
    connection.setTcpNoDelay(true);
  }

  private void closeConnection() {
    if (connection != null) {
      try {
        connection.close();
      } catch (IOException ignored) {
      }
    }
    connection = null;
    sos = null;
    sis = null;
    readAvail = 0;
  }

  @Override
  public void run() {
    // reader thread
    while (true) {

      while (connection == null) {
        try {
          synchronized (this) {
            if (endpoint == null) {
              return;
            }
            connect();
          }
          delegate.onReconnect();
        } catch (TransportException e) {
          try {
            // TODO: figure out a back-off strategy
            Thread.sleep(config.pauseBeforeReconnect.toMillis());
          } catch (InterruptedException ignored) {
          }
        }
      }

      try {
        int pos = readFrom(sis, 2);
        byte[] hdr = getReadBuffer();

        final int op = 0x7F & hdr[pos + 0];
        final boolean fin = hdr[pos + 0] < 0;
        int len = 0X7F & hdr[pos + 1];
        final boolean mask = hdr[pos + 1] < 0;
        if (mask || !fin) {
          throw new IOException("WebSocket frame invalid");
        }
        int headerLength = 0;
        if (len == 126) {
          headerLength = 2;
        } else if (len == 127) {
          headerLength = 8;
        }

        if (headerLength > 0) {
          pos = readFrom(sis, headerLength);
          hdr = getReadBuffer();
          if (len == 126) {
            len = (0XFF & hdr[pos + 0]) << 8 | 0XFF & hdr[pos + 1];
          } else /*if (len == 127)*/ {
            if ((hdr[pos + 0] | hdr[pos + 1] | hdr[pos + 2] | hdr[pos + 3]) != 0) {
              throw new IOException("WebSocket message length invalid");
            }
            len = (0XFF & hdr[pos + 4]) << 24 | (0XFF & hdr[pos + 5]) << 16 | (0XFF & hdr[pos + 6]) << 8 | 0XFF & hdr[pos + 7];
          }
        }

        if (len > 125 && (op < 1 || op > 2)) {
          throw new IOException("WebSocket frame invalid");
        }

        if (len > 1024 * 1024) {
          throw new IOException("WebSocket frame too big " + len + "> 1MB");
        }

        int offset = readFrom(sis, len);

        if (op == 1 || op == 2) {
          delegate.onMessage(getReadBuffer(), offset, len);
        } else if (op == 8) {
          // TODO close
        } else if (op == 9) {
          // TODO ping
        }
      } catch (IOException e) {
        log.info("connection closed");
        synchronized (this) {
          closeConnection();
        }
        delegate.onDisconnect();
      }
    }
  }

  private byte[] readBuffer = new byte[65536];
  private int readAvail = 0;
  private int readPos = 0;

  private byte[] getReadBuffer() {
    return readBuffer;
  }

  private int readFrom(InputStream sis, int length) throws IOException {
    if (length <= readAvail) {
      int retVal = readPos;
      readAvail -= length;
      readPos += length;
      return retVal;
    }
    if (readBuffer.length < length) {
      // should rarely happen
      byte[] newBuf = new byte[length];
      System.arraycopy(readBuffer, readPos, newBuf, 0, readAvail);
      readBuffer = newBuf;
      readPos = 0;
    }
    if (readAvail == 0)
      readPos = 0;
    int avail = sis.available();
    if (avail + readAvail > readBuffer.length) {
      avail = readBuffer.length - readAvail;
    } else if (avail + readAvail < length) {
      avail = length - readAvail;
    }
    if (avail + readPos + readAvail > readBuffer.length) {
      System.arraycopy(readBuffer, readPos, readBuffer, 0, readAvail);
      readPos = 0;
    }
    int retVal = readPos;
    while (avail > 0) {
      int nRead = sis.read(readBuffer, readPos + readAvail, avail);
      if (nRead < 0) {
        throw new IOException("EOF reached");
      }
      readAvail += nRead;
      avail -= nRead;
    }
    readPos += length;
    readAvail -= length;
    return retVal;
  }

  private static final String SDK_USER_AGENT;

  static {
    String version = "invalid";
    try (InputStream input = SyncTransport.class.getClassLoader().getResourceAsStream("driveline-sdk.properties")) {
      Properties properties = new Properties();
      properties.load(input);
      version = properties.getProperty("version");
    } catch (NullPointerException | IOException e) {
      log.error("cannot load properties", e);
    }
    SDK_USER_AGENT = "driveline/" + version + " java";
  }

  private static final byte[] HTTP_HEADER = (
    "GET /control HTTP/1.1\r\n"
      + "Host: 1533.io\r\n"
      + "Upgrade: websocket\r\n"
      + "Connection: Upgrade\r\n"
      + "Sec-WebSocket-Key: dGhlIHNhbXBsZSBub25jZQ==\r\n"
      + "Sec-WebSocket-Version: 13\r\n"
      + "Sec-WebSocket-Protocol: driveline\r\n"
      + "User-Agent: " + SDK_USER_AGENT + "\r\n"
      + "\r\n"
  ).getBytes();

  private static final byte[] HTTP_RESP = "HTTP/1.1 101 Switching Protocols\r\n".getBytes();
}
