/*
 * Copyright 2019, 1533 Systems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package driveline.protocol;

import java.time.Duration;
import java.time.temporal.TemporalUnit;

import driveline.cbor.encoder.CborEncoder;

public final class StoreOptions extends CommandOptions {
  private long ttl;
  private byte[] casID;

  public StoreOptions() {
    this.ttl = -1;
    this.casID = null;
  }

  public StoreOptions withTTL(long durationInMillisecond) {
    this.ttl = durationInMillisecond;
    return this;
  }

  public StoreOptions withTTL(long value, TemporalUnit unit) {
    this.ttl = Duration.of(value, unit).toMillis();
    return this;
  }

  public StoreOptions withTTL(Duration value) {
    this.ttl = value.toMillis();
    return this;
  }

  public StoreOptions withCasId(byte[] recordId) {
    this.casID = recordId;
    return this;
  }

  @Override
  public void encode(CborEncoder encoder) {
    if (ttl == -1 || casID == null) {
      encoder.encodeNull();
      return;
    }

    encoder.encodeArray();
    if (ttl != -1) {
      encoder.encode(TagId.STORE_TTL).encode(ttl);
    }
    if (casID != null) {
      encoder.encode(TagId.STORE_CAS_ID).encode(casID);
    }
    encoder.endArray();
  }
}
