/*
 * Copyright 2019, 1533 Systems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package driveline.protocol;

import driveline.cbor.Cbor;
import driveline.cbor.CborException;
import driveline.cbor.decoder.CborBuffer;

@SuppressWarnings("Duplicates")
public abstract class ServerMessage {

  public final long consumerID;

  ServerMessage(long consumerID) {
    this.consumerID = consumerID;
  }

  public static ServerMessage fromBytes(byte[] b, int offset, int length) throws CborException {
    return fromBytes(new ServerMessageDecoder(), b, offset, length);
  }

  public static ServerMessage fromBytes(ServerMessageDecoder decoder, byte[] b, int offset, int length) throws CborException {
    decoder.reset(b, offset, length);
    return decoder.getServerMessage();
  }

  public static final class ServerMessageDecoder extends CborBuffer {

    public ServerMessageDecoder() {
      super();
    }

    ServerMessageDecoder(byte[] buf, int offset, int length) {
      super(buf, offset, length);
    }

    final ServerMessage getServerMessage() throws CborException {
      if ((Cbor.TypeMask & buf[offset]) != Cbor.Array) {
        throw new CborException(INVALID_SERVER_MESSAGE);
      }
      int messageCount = getVarInt16() - 3;
      if (buf[offset] == (Cbor.TextString | 4) && buf[offset + 1] == (byte) 'd') {
        offset += 5;
        return dataMessageFromBytes(messageCount);
      } else if (buf[offset] == (Cbor.TextString | 3)) {
        byte c = buf[offset + 1];
        if (c == (byte) 'e') {
          offset += 4;
          return errorMessageFromBytes();
        } else if (c == (byte) 's') {
          offset += 4;
          return syncMessageFromBytes();
        }
      }
      throw new CborException(INVALID_SERVER_MESSAGE);
    }

    final DataMessage dataMessageFromBytes(int messageCount) throws CborException {
      if ((Cbor.TypeMask & buf[offset]) != Cbor.UnsignedInteger) {
        throw new CborException(INVALID_SERVER_MESSAGE);
      }
      long consumerId = getVarInt64();

      // prepare for tag decoding
      byte[][] messageIds = null;

      if ((Cbor.TypeMask & buf[offset]) == Cbor.Array) {
        int tagCount = getVarInt8();
        for (int i = 0; i < tagCount; i += 2) {
          int tagId = getVarInt8();
          if (tagId == TagId.ID) {
            if ((Cbor.TypeMask & buf[offset]) != Cbor.Array) {
              throw new CborException(INVALID_SERVER_MESSAGE);
            }
            int idCount = getVarInt16();
            messageIds = new byte[idCount][];
            for (int j = 0; j < idCount; j++) {
              messageIds[j] = getByteString();
            }
          } else {
            throw new CborException(INVALID_SERVER_MESSAGE);
          }
        }
      } else if (isBlank()) {
        offset++;
      } else {
        throw new CborException(INVALID_SERVER_MESSAGE);
      }
      byte[][] messages = new byte[messageCount][];
      for (int i = 0; i < messageCount; i++) {
        if (isBlank()) {
          offset++;
          messages[i] = null;
        } else {
          messages[i] = getByteString();
        }
      }
      return new DataMessage(consumerId, messages, messageIds);
    }

    final ErrorMessage errorMessageFromBytes() throws CborException {
      if ((Cbor.TypeMask & buf[offset]) != Cbor.UnsignedInteger) {
        throw new CborException(INVALID_SERVER_MESSAGE);
      }
      long consumerId = getVarInt64();
      String message = getTextString();
      return new ErrorMessage(consumerId, message);
    }

    final SyncMessage syncMessageFromBytes() throws CborException {
      if ((Cbor.TypeMask & buf[offset]) != Cbor.UnsignedInteger) {
        throw new CborException(INVALID_SERVER_MESSAGE);
      }
      long consumerId = getVarInt64();
      return new SyncMessage(consumerId);
    }

    private boolean isBlank() {
      int current = 0xFF & buf[offset];
      return current == Cbor.NULL || current == Cbor.UNDEFINED;
    }
  }

  public abstract boolean isOK();

  private static final String INVALID_SERVER_MESSAGE = "invalid server message";
}
