/*
 * Copyright 2019, 1533 Systems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package driveline.cbor.encoder;

import driveline.cbor.Cbor;

abstract class CborDataItem {

  abstract int encodedSize();

  abstract int encode(byte[] buf, int offset);

  static int encodeTypeAndSize(byte[] buf, int offset, int type, long size) {
    if (size < 24L) {
      buf[offset] = (byte) (type | size);
      return 1;
    } else if (size < 0x100L) {
      buf[offset] = (byte) (type | 24);
      buf[offset + 1] = (byte) size;
      return 2;
    } else if (size < 0x10000L) {
      buf[offset] = (byte) (type | 25);
      buf[offset + 1] = (byte) (size >>> 8);
      buf[offset + 2] = (byte) (size & 0xFF);
      return 3;
    } else if (size < 0x100000000L) {
      buf[offset] = (byte) (type | 26);
      buf[offset + 1] = (byte) ((size >>> 24) & 0xFF);
      buf[offset + 2] = (byte) ((size >>> 16) & 0xFF);
      buf[offset + 3] = (byte) ((size >>> 8) & 0xFF);
      buf[offset + 4] = (byte) (size & 0xFF);
      return 5;
    } else {
      buf[offset] = (byte) (type | 27);
      buf[offset + 1] = (byte) ((size >>> 56) & 0xFF);
      buf[offset + 2] = (byte) ((size >>> 48) & 0xFF);
      buf[offset + 3] = (byte) ((size >>> 40) & 0xFF);
      buf[offset + 4] = (byte) ((size >>> 32) & 0xFF);
      buf[offset + 5] = (byte) ((size >>> 24) & 0xFF);
      buf[offset + 6] = (byte) ((size >>> 16) & 0xFF);
      buf[offset + 7] = (byte) ((size >>> 8) & 0xFF);
      buf[offset + 8] = (byte) (size & 0xFF);
      return 9;
    }
  }

  static int encodedLength(long value) {
    if (value < 24L) {
      return 1;
    } else if (value < 0x100L) {
      return 2;
    } else if (value < 0x10000L) {
      return 3;
    } else if (value < 0x100000000L) {
      return 5;
    } else {
      return 9;
    }
  }

  static final CborDataItem Null = byteValueItem(Cbor.NULL);
  static final CborDataItem Undefined = byteValueItem(Cbor.UNDEFINED);
  static final CborDataItem True = byteValueItem(Cbor.TRUE);
  static final CborDataItem False = byteValueItem(Cbor.FALSE);

  private static CborDataItem byteValueItem(int type) {

    return new CborDataItem() {
      @Override
      int encodedSize() {
        return 1;
      }

      @Override
      int encode(byte[] buf, int offset) {
        buf[offset] = (byte) type;
        return 1;
      }
    };
  }

}
