/*
 * Copyright 2019, 1533 Systems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package driveline;

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import driveline.cbor.Cbor;
import driveline.cbor.CborException;
import driveline.cbor.decoder.CborBuffer;
import driveline.protocol.DataMessage;
import driveline.protocol.ErrorMessage;
import driveline.protocol.ListOptions;

final class ListConsumer extends Consumer<Iterable<String>, ListOptions> {

  private static final Logger log = LoggerFactory.getLogger(ListConsumer.class);
  private final ListDecoder decoder = new ListDecoder();
  private final LinkedBlockingQueue<Result<Exception, String>> outputQueue;
  final String pattern;
  private final ListOptions options;
  private boolean isDone;

  ListConsumer(DrivelineClient client, int id, String pattern, ListOptions options) {
    super(client, id);
    this.outputQueue = new LinkedBlockingQueue<>();
    this.result = new ListResult(outputQueue);

    this.isDone = false;
    this.pattern = pattern;
    this.options = options;
  }

  static final class ListDecoder extends CborBuffer {

    String[] getStreamList() throws CborException {
      if ((Cbor.TypeMask & buf[offset]) != Cbor.Array) {
        throw new CborException("Invalid message");
      }
      int entryCount = getVarInt32();
      String[] result = new String[entryCount];
      for (int i = 0; i < entryCount; i++) {
        result[i] = getTextString();
      }
      return result;
    }
  }

  @Override

  public Consumer handleMessage(DataMessage msg) {
    // FIXME geraud: a tad defensive.
    if (isDone) {
      return null;
    }
    try {
      try {
        decoder.reset(msg.messages[0], 0, msg.messages[0].length);
        String[] entries = decoder.getStreamList();
        int count = entries.length;
        if (count == 0) {
          isDone = true;
          outputQueue.put(Result.failure(new StopIteration()));
          return null;
        }
        for (int i = 0; i < count; i++) {
          String entry = entries[i];
          outputQueue.put(Result.ok(entry));
        }
        return this;
      } catch (CborException e) {
        outputQueue.put(Result.failure(new DrivelineException("invalid stream name received")));
      }
    } catch (InterruptedException e) {
      log.warn("interrupted while adding error to list result");
    }
    isDone = true;
    return null;
  }

  @Override
  Consumer handleError(ErrorMessage msg) {
    // FIXME geraud: a tad defensive.
    if (isDone) {
      return null;
    }
    isDone = true;
    try {
      outputQueue.put(Result.failure(new DrivelineException(msg.message)));
    } catch (InterruptedException e) {
      log.warn("interrupted while adding error to list result");
    }
    return null;
  }

  @Override
  ListOptions getOptions() {
    return options;
  }

  @Override
  Consumer onDisconnect() {
    isDone = true;
    try {
      outputQueue.put(Result.failure(new DrivelineException("connection lost")));
    } catch (InterruptedException e) {
      log.warn("interrupted while adding error to list result");
    }
    return null;
  }

  @Override
  Consumer onReconnect() {
    return null;
  }

  private static class ListResult implements Iterable<String>, Iterator<String> {

    private final BlockingQueue<Result<Exception, String>> list;
    private boolean isDone;
    private Result<Exception, String> next;

    ListResult(BlockingQueue<Result<Exception, String>> list) {
      this.list = list;
      this.isDone = false;
      this.next = null;
    }

    @Override
    public boolean hasNext() {
      if (isDone) {
        return false;
      }
      try {
        if (next == null) {
          next = list.take();
        }
        if (next.isOK()) {
          return true;
        } else {
          if (next.getFailure() instanceof StopIteration) {
            isDone = true;
            next = null;
            return false;
          } else {
            return true;
          }
        }
      } catch (InterruptedException e) {
        isDone = true;
        next = null;
        return false;
      }
    }

    @Override
    public String next() {
      if (hasNext()) {
        if (next.isOK()) {
          String result = this.next.getResult();
          next = null;
          return result;
        }
        throw new RuntimeException(next.getFailure());
      }
      throw new NoSuchElementException();
    }

    @SuppressWarnings("NullableProblems")
    @Override
    public Iterator<String> iterator() {
      return this;
    }

  }

  private static final class StopIteration extends Exception {
  }
}

