/*
 * Copyright 2019, 1533 Systems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package driveline.cbor.encoder;

import java.nio.charset.Charset;

import driveline.cbor.Cbor;

@SuppressWarnings({"WeakerAccess", "unused", "UnusedReturnValue"})
public class CborEncoder {
  private static final Charset UTF8 = Charset.forName("UTF-8");

  private CborDataItem[] items = new CborDataItem[8];
  private int itemCount = 0;

  private int[] containers = new int[8];
  private int containerCount = 0;

  private CborDataItemContainer lastContainer = null;

  public CborEncoder encode(CborSerializable item) {
    item.encode(this);
    return this;
  }

  public CborEncoder encodeNullable(CborSerializable item) {
    return item == null ? encodeNull() : encode(item);
  }

  public CborEncoder encodeArray() {
    addDataItemContainer(new CborDataItemContainer(Cbor.Array));
    return this;
  }

  public CborEncoder endArray() {
    return endContainer(Cbor.Array);
  }

  public CborEncoder encodeMap() {
    addDataItemContainer(new CborDataItemContainer(Cbor.Map));
    return this;
  }

  public CborEncoder endMap() {
    return endContainer(Cbor.Map);
  }

  CborEncoder endContainer(int type) {
    if (lastContainer == null || lastContainer.type != type) {
      throw new IllegalStateException("cannot end '" + (type == Cbor.Array ? "array" : "map") + "': no such container");
    }
    if (lastContainer.type == Cbor.Map && lastContainer.itemCount % 2 != 0) {
      throw new IllegalStateException("map does not contain an even number of items (" + lastContainer.itemCount + ")");
    }
    containerCount--;
    lastContainer = containerCount == 0 ? null : (CborDataItemContainer) items[containers[containerCount - 1]];
    return this;
  }

  public CborEncoder encode(String value) {
    addDataItem(new CborData(Cbor.TextString, value.getBytes(UTF8)));
    return this;
  }

  public CborEncoder encode(byte[] value) {
    addDataItem(new CborData(Cbor.ByteString, value));
    return this;
  }

  public CborEncoder encode(long value) {
    addDataItem(new CborInteger(value));
    return this;
  }

  public CborEncoder encode(boolean value) {
    addDataItem(value ? CborDataItem.True : CborDataItem.False);
    return this;
  }

  public CborEncoder encodeNull() {
    addDataItem(CborDataItem.Null);
    return this;
  }

  public CborEncoder encodeUndefined() {
    addDataItem(CborDataItem.Undefined);
    return this;
  }

  public byte[] getBytes() {
    int sizeInBytes = 0;
    for (int i = 0; i < itemCount; i++) {
      sizeInBytes += items[i].encodedSize();
    }
    byte[] encodedData = new byte[sizeInBytes];
    int offset = 0;
    for (int i = 0; i < itemCount; i++) {
      offset += items[i].encode(encodedData, offset);
    }
    return encodedData;
  }

  int addDataItem(CborDataItem item) {
    if (lastContainer != null) {
      lastContainer.incrementItemCount();
    }
    if (itemCount >= items.length) {
      CborDataItem[] oldItems = items;
      this.items = new CborDataItem[oldItems.length * 2];
      System.arraycopy(oldItems, 0, items, 0, itemCount);
    }
    items[itemCount++] = item;
    return itemCount - 1;
  }

  int addDataItemContainer(CborDataItemContainer container) {
    int itemIndex = addDataItem(container);
    if (containerCount >= containers.length) {
      int[] oldContainers = containers;
      this.containers = new int[containers.length * 2];
      System.arraycopy(oldContainers, 0, containers, 0, containerCount);
    }
    containers[containerCount++] = itemIndex;
    lastContainer = container;
    return itemIndex;
  }

  public static CborEncoder arrayEncoder() {
    return new CborEncoder().encodeArray();
  }

  public static CborEncoder mapEncoder() {
    return new CborEncoder().encodeMap();
  }

}
