/*
 * Copyright 2019, 1533 Systems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package driveline.cbor.decoder;

import java.nio.charset.Charset;

import driveline.cbor.Cbor;
import driveline.cbor.CborException;

@SuppressWarnings("WeakerAccess")
public class CborBuffer {
  public byte[] buf;
  public int start;
  public int length;
  public int offset;

  public CborBuffer() {
    buf = null;
    start = 0;
    length = 0;
    offset = 0;
  }

  public CborBuffer(byte[] buf, int offset, int length) {
    reset(buf, offset, length);
  }

  public final void reset(byte[] buf, int offset, int length) {
    this.buf = buf;
    this.start = offset;
    this.offset = offset;
    this.length = length;
  }



  public int getVarInt8() throws CborException {
    int code = Cbor.SizeMask & buf[offset++];
    if (code < 24) {
      return code;
    } else if (code == 24) {
      return 0xFF & buf[offset++];
    }
    throw new CborException(INVALID_SERVER_MESSAGE);
  }

  public int getVarInt16() throws CborException {
    int code = Cbor.SizeMask & buf[offset++];
    if (code < 24) {
      return code;
    } else if (code == 24) {
      return (0xFF & buf[offset++]);
    } else if (code == 25) {
      return (0xFF & buf[offset++]) << 8 | (0xFF & buf[offset++]);
    }
    throw new CborException(INVALID_SERVER_MESSAGE);
  }

  public int getVarInt32() throws CborException {
    int code = Cbor.SizeMask & buf[offset++];
    if (code < 24) {
      return code;
    } else if (code == 24) {
      return (0xFF & buf[offset++]);
    } else if (code == 25) {
      return (0xFF & buf[offset++]) << 8 | (0xFF & buf[offset++]);
    } else if (code == 26) {
      return (0xFF & buf[offset++]) << 24 | (0xFF & buf[offset++]) << 16 | (0xFF & buf[offset++]) << 8 | (0xFF & buf[offset++]);
    }
    throw new CborException(INVALID_SERVER_MESSAGE);
  }

  public long getVarInt64() throws CborException {
    long code = Cbor.SizeMask & buf[offset++];
    if (code < 24) {
      return code;
    } else if (code == 24) {
      return (0xFFL & buf[offset++]);
    } else if (code == 25) {
      return (0xFFL & buf[offset++]) << 8 | (0xFFL & buf[offset++]);
    } else if (code == 26) {
      return (0xFFL & buf[offset++]) << 24 | (0xFFL & buf[offset++]) << 16 | (0xFFL & buf[offset++]) << 8 | (0xFFL & buf[offset++]);
    } else if (code == 27) {
      return (0xFFL & buf[offset++]) << 56 | (0xFFL & buf[offset++]) << 48 | (0xFFL & buf[offset++]) << 40 | (0xFFL & buf[offset++]) << 32
        | (0xFFL & buf[offset++]) << 24 | (0xFFL & buf[offset++]) << 16 | (0xFFL & buf[offset++]) << 8 | (0xFFL & buf[offset++]);
    }
    throw new CborException(INVALID_SERVER_MESSAGE);
  }

  public String getTextString() throws CborException {
    if ((Cbor.TypeMask & buf[offset]) != Cbor.TextString) {
      throw new CborException(INVALID_SERVER_MESSAGE);
    }
    long length = getVarInt32();
    String result = new String(buf, offset, (int) length, UTF8);
    offset += length;
    return result;
  }

  public byte[] getByteString() throws CborException {
    if ((Cbor.TypeMask & buf[offset]) != Cbor.ByteString) {
      throw new CborException(INVALID_SERVER_MESSAGE);
    }
    int length = getVarInt32();
    byte[] result = new byte[length];
    System.arraycopy(buf, offset, result, 0, length);
    offset += length;
    return result;
  }

  private static final String INVALID_SERVER_MESSAGE = "Invalid server message";
  private static final Charset UTF8 = Charset.forName("UTF-8");

}

