/*
 * Copyright 2019, 1533 Systems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package driveline;

import java.util.HashMap;
import java.util.Map;

import driveline.cbor.encoder.CborEncoder;
import driveline.cbor.encoder.CborSerializable;

interface StreamId extends CborSerializable {

  default boolean isAlias() {
    return false;
  }

  static StreamId of(String streamName) {
    return new TextualStreamId(streamName);
  }

  static StreamId of(int alias) {
    return new NumericStreamId(alias);
  }

  class NumericStreamId implements StreamId {
    final int streamId;

    NumericStreamId(int streamId) {
      this.streamId = streamId;
    }

    @Override
    public boolean isAlias() {
      return true;
    }

    @Override
    public void encode(CborEncoder encoder) {
      encoder.encode(streamId);
    }

    @Override
    public boolean equals(Object obj) {
      if (obj instanceof NumericStreamId) {
        return streamId == ((NumericStreamId) obj).streamId;
      } else {
        return false;
      }
    }

    @Override
    public int hashCode() {
      return Integer.hashCode(streamId);
    }
  }

  class TextualStreamId implements StreamId {
    final String streamId;

    TextualStreamId(String streamId) {
      this.streamId = streamId;
    }

    @Override
    public void encode(CborEncoder encoder) {
      encoder.encode(streamId);
    }

    @Override
    public boolean equals(Object obj) {
      if (obj instanceof TextualStreamId) {
        return streamId.equals(((TextualStreamId) obj).streamId);
      } else {
        return false;
      }
    }

    @Override
    public int hashCode() {
      return streamId.hashCode();
    }
  }

  final class Factory {
    private final int[] stack;
    private int freeAliases;
    private final Map<Integer, String> aliases;

    Factory(int capacity) {
      if (capacity < 0) {
        throw new IllegalArgumentException("StreamId.Factory must be initialized with a positive number of values");
      }
      stack = new int[capacity];
      for (int i = 0; i < capacity; i++) {
        stack[i] = capacity - 1 - i;
      }
      aliases = new HashMap<>(capacity);
      freeAliases = capacity;
    }

    final StreamId get(String streamName) {
      synchronized (aliases) {
        if (freeAliases == 0) {
          return StreamId.of(streamName);
        } else {
          int id = stack[--freeAliases];
          aliases.put(id, streamName);
          return StreamId.of(id);
        }
      }
    }

    final void release(StreamId streamId) {
      if (streamId.isAlias()) {
        synchronized (aliases) {
          int id = ((NumericStreamId) streamId).streamId;
          aliases.remove(id);
          stack[freeAliases++] = id;
        }
      }
    }

    final Map<Integer, String> getAliases() {
      synchronized (aliases) {
        return new HashMap<>(aliases);
      }
    }

    final int available() {
      return freeAliases;
    }
  }
}
