/*
 * Copyright 2019, 1533 Systems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package driveline;

import java.util.Objects;
import java.util.concurrent.*;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import driveline.protocol.DataMessage;
import driveline.protocol.ErrorMessage;
import driveline.protocol.QueryOptions;

@SuppressWarnings("unused")
public final class Query extends Consumer<Query, QueryOptions> {
  private static final Logger log = LoggerFactory.getLogger(Query.class);

  final String query;
  private final RecordHandler handler;
  final QueryOptions options;
  private final ThreadPoolExecutor executor;
  private byte[] lastMessageId;
  public final boolean isContinuous;

  Query(DrivelineClient client, int id, String query, boolean isContinuous, QueryOptions options, RecordHandler handler) {
    super(client, id);
    Objects.requireNonNull(query);
    Objects.requireNonNull(options);
    Objects.requireNonNull(handler);
    this.query = query;
    this.isContinuous = isContinuous;
    this.handler = handler;
    this.options = options;
    this.result = this;
    if (options.getConcurrency() == 0) {
      this.executor = null;
    } else {
      this.executor = new ThreadPoolExecutor(
        0, options.getConcurrency(),
        10L, TimeUnit.SECONDS,
        new LinkedBlockingQueue<>(options.getConcurrency() * 16)
      );
      this.executor.setRejectedExecutionHandler((runnable, executor) -> {
        try {
          executor.getQueue().put(runnable);
        } catch (InterruptedException e) {
          Thread.currentThread().interrupt();
          log.error("Interrupted while queuing messages ", e);
        }
      });
    }
  }

  @Override
  Consumer handleMessage(DataMessage msg) {
    byte[][] messageIds = msg.getMessageIDs();
    byte[][] messages = msg.messages;
    int count = messages.length;

    if (executor == null) {
      if (messageIds != null) {
        byte[] messageId;
        byte[] message;
        for (int i = 0; i < count; i++) {
          messageId = messageIds[i];
          this.lastMessageId = messageId;
          handler.onRecord(new DataRecord(messages[i], messageId));
        }
      } else {
        for (int i = 0; i < count; i++) {
          handler.onRecord(new DataRecord(messages[i], null));
        }
      }
    } else {
      if (messageIds != null) {
        for (int i = 0; i < count; i++) {
          byte[] message = messages[i];
          byte[] messageId = messageIds[i];
          this.lastMessageId = messageId;
          executor.execute(() -> handler.onRecord(new DataRecord(message, messageId)));
        }
      } else {
        for (int i = 0; i < count; i++) {
          byte[] message = messages[i];
          executor.execute(() -> handler.onRecord(new DataRecord(message, null)));
        }
      }
    }
    return this;
  }

  @Override
  Consumer handleError(ErrorMessage msg) {
    handler.onRecord(new ErrorRecord(msg.message));
    return null;
  }

  @Override
  public void close() throws DrivelineException {
    client.cancel(this);
  }

  @Override
  Consumer onReconnect() throws DrivelineException {
    if (lastMessageId != null) {
      options.fromMessage(lastMessageId);
    }
    client.query(this);
    return this;
  }

  @Override
  Consumer onDisconnect() {
    return this;
  }

  @Override
  QueryOptions getOptions() {
    return options;
  }
}
