/*
 * Copyright 2019, 1533 Systems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package driveline;

import java.net.URI;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import driveline.cbor.encoder.CborEncoder;
import driveline.protocol.AppendOptions;
import driveline.protocol.CancelOptions;
import driveline.protocol.ListOptions;
import driveline.protocol.LoadOptions;
import driveline.protocol.QueryOptions;
import driveline.protocol.ServerMessage;
import driveline.protocol.StoreOptions;
import driveline.transport.SyncTransport;
import driveline.transport.Transport;
import driveline.transport.TransportConfig;
import driveline.transport.TransportDelegate;
import driveline.transport.TransportException;

import static driveline.protocol.ServerMessage.ServerMessageDecoder;

@SuppressWarnings({"WeakerAccess", "unused"})
public class DrivelineClient implements TransportDelegate {
  private static final Logger log = LoggerFactory.getLogger(DrivelineClient.class);
  private static final int MAX_ALIASES = 256;
  private static final String ERR_COMMAND_FAILED = "cannot send command";

  private static final String SYNC = "syn";

  private static final String CONTINUOUS_QUERY = "sq";
  private static final String QUICK_QUERY = "qq";
  private static final String CANCEL = "can";
  private static final String DEFINE = "def";

  private static final String LIST_KV = "lst";
  private static final String STORE_KV = "st";
  private static final String LOAD_KV = "ld";
  private static final String REMOVE_KV = "rm";
  private static final String REMOVE_MATCHES_KV = "rmk";

  private static final String APPEND_STREAM = "app";
  private static final String LIST_STREAM = "sls";
  private static final String TRUNCATE_STREAM = "trc";

  private final URI endpoint;
  private final Transport transport;
  final AtomicInteger nextConsumerId = new AtomicInteger();
  final StreamId.Factory streamIdFactory = new StreamId.Factory(MAX_ALIASES);
  private final Map<Long, Consumer> consumers = new HashMap<>();
  private final ServerMessageDecoder decoder = new ServerMessageDecoder();

  private DrivelineClient(URI endpoint, Transport transport) {
    this.endpoint = endpoint;
    this.transport = transport;
  }

  public void start() throws DrivelineException {
    start(TransportConfig.getDefault());
  }

  public void start(TransportConfig config) throws DrivelineException {
    try {
      transport.connect(endpoint, config, this);
    } catch (TransportException e) {
      throw new DrivelineException("cannot start client", e);
    }
  }

  public void stop() throws DrivelineException {
    try {
      transport.disconnect();
    } catch (TransportException e) {
      throw new DrivelineException("cannot stop client", e);
    }
  }

  public Query continuousQuery(String dql, RecordHandler handler) throws DrivelineException {
    return continuousQuery(dql, new QueryOptions(), handler);
  }

  public Query continuousQuery(String dql, QueryOptions options, RecordHandler handler) throws DrivelineException {
    return query(new Query(this, nextConsumerID(), dql, true, options, handler));
  }

  public Query query(String dql, RecordHandler handler) throws DrivelineException {
    return query(dql, new QueryOptions(), handler);
  }

  public Query query(String dql, QueryOptions options, RecordHandler handler) throws DrivelineException {
    return query(new Query(this, nextConsumerID(), dql, false, options, handler));
  }

  Query query(Query query) throws DrivelineException {
    registerConsumer(query);
    String command = query.isContinuous ? CONTINUOUS_QUERY : QUICK_QUERY;
    sendCommand(CborEncoder.arrayEncoder()
      .encode(command).encode(query.consumerId).encode(query.options).encode(query.query)
      .getBytes());
    return query.getResult();
  }

  public void cancel(Query query) throws DrivelineException {
    cancel(query, new CancelOptions());
  }

  public void cancel(Query query, CancelOptions options) throws DrivelineException {
    Objects.requireNonNull(query);
    Objects.requireNonNull(options);
    if (!consumers.containsKey(query.consumerId)) {
      return;
    }
    try {
      sendCommand(CborEncoder.arrayEncoder()
        .encode(CANCEL).encode(query.consumerId).encode(options)
        .getBytes());
    } finally {
      unregisterConsumer(query);
    }
  }

  public Stream openStream(String stream) throws DrivelineException {
    Objects.requireNonNull(stream);
    StreamId streamId = streamIdFactory.get(stream);
    if (streamId.isAlias()) {
      define(streamId, stream);
    }
    return new Stream(this, streamId);
  }

  public void closeStream(Stream stream) {
    streamIdFactory.release(stream.streamId);
  }

  public void append(String stream, byte[] record) throws DrivelineException {
    append(StreamId.of(stream), record, new AppendOptions());
  }

  public void append(String stream, byte[] record, AppendOptions options) throws DrivelineException {
    append(StreamId.of(stream), record, options);
  }

  public void append(String stream, Collection<byte[]> records) throws DrivelineException {
    append(stream, records, new AppendOptions());
  }

  public void append(String stream, Collection<byte[]> records, AppendOptions options) throws DrivelineException {
    StreamId streamId = StreamId.of(stream);
    for (byte[] record : records) {
      append(streamId, record, options);
    }
  }

  void append(StreamId streamId, byte[] record, AppendOptions options) throws DrivelineException {
    sendCommand(CborEncoder.arrayEncoder()
      .encode(APPEND_STREAM).encode(streamId).encode(options).encode(record)
      .getBytes());
  }

  public void truncate(String stream) throws DrivelineException {
    truncate(StreamId.of(stream));
  }

  void truncate(StreamId streamId) throws DrivelineException {
    sendCommand(CborEncoder.arrayEncoder()
      .encode(TRUNCATE_STREAM).encodeNull().encode(streamId)
      .getBytes());
  }

  public Iterable<String> listStreams(String streamPattern) throws DrivelineException {
    return list(LIST_STREAM, streamPattern, new ListOptions());
  }

  public Iterable<String> listStreams(String streamPattern, ListOptions options) throws DrivelineException {
    return list(LIST_STREAM, streamPattern, options);
  }

  public void store(String keyName, byte[] record) throws DrivelineException {
    store(keyName, record, new StoreOptions());
  }

  public void store(String keyName, byte[] record, StoreOptions options) throws DrivelineException {
    sendCommand(CborEncoder.arrayEncoder()
      .encode(STORE_KV).encode(keyName).encode(options).encode(record)
      .getBytes());
  }

  public CompletableFuture<Record> load(String key) throws DrivelineException {
    return load(key, new LoadOptions());
  }

  public CompletableFuture<Record> load(String key, LoadOptions options) throws DrivelineException {
    return load(new LoadConsumer(this, nextConsumerID(), key, options));
  }

  CompletableFuture<Record> load(LoadConsumer consumer) throws DrivelineException {
    registerConsumer(consumer);
    sendCommand(CborEncoder.arrayEncoder()
      .encode(LOAD_KV).encode(consumer.consumerId).encode(consumer.getOptions()).encode(consumer.keyName)
      .getBytes());
    return consumer.getResult();

  }

  public void remove(String key) throws DrivelineException {
    sendCommand(CborEncoder.arrayEncoder()
      .encode(REMOVE_KV).encodeUndefined().encode(key)
      .getBytes());
  }

  public void removeMatches(String keyPattern) throws DrivelineException {
    sendCommand(CborEncoder.arrayEncoder()
      .encode(REMOVE_MATCHES_KV).encodeUndefined().encode(keyPattern)
      .getBytes());
  }

  public Iterable<String> listKeys(String keyPattern) throws DrivelineException {
    return list(LIST_KV, keyPattern, new ListOptions());
  }

  public Iterable<String> listKeys(String keyPattern, ListOptions options) throws DrivelineException {
    return list(LIST_KV, keyPattern, options);
  }

  private Iterable<String> list(String command, String pattern, ListOptions options) throws DrivelineException {
    ListConsumer consumer = new ListConsumer(this, nextConsumerID(), pattern, options);
    registerConsumer(consumer);
    sendCommand(CborEncoder.arrayEncoder()
      .encode(command).encode(consumer.consumerId).encode(consumer.getOptions()).encode(consumer.pattern)
      .getBytes());
    return consumer.getResult();
  }

  public CompletableFuture<Void> sync() throws DrivelineException {
    SyncConsumer consumer = new SyncConsumer(this, nextConsumerID());
    registerConsumer(consumer);
    sendCommand(CborEncoder.arrayEncoder()
      .encode(SYNC).encode(consumer.consumerId)
      .getBytes());
    return consumer.getResult();
  }

  void define(StreamId streamId, String stream) throws DrivelineException {
    sendCommand(CborEncoder.arrayEncoder().encode(DEFINE).encode(streamId).encode(stream).getBytes());
  }

  private void registerConsumer(Consumer c) {
    synchronized (consumers) {
      consumers.put(c.consumerId, c);
    }
  }

  private void unregisterConsumer(Consumer consumer) {
    synchronized (consumers) {
      consumers.remove(consumer.consumerId);
    }
  }

  private void sendCommand(byte[] command) throws DrivelineException {
    try {
      transport.send(command);
    } catch (TransportException e) {
      throw new DrivelineException(ERR_COMMAND_FAILED, e);
    }
  }

  public void onConnect() {
  }

  public void onReconnect() {
    try {
      Map<Integer, String> aliases = streamIdFactory.getAliases();
      for (Map.Entry<Integer, String> entry : aliases.entrySet()) {
        define(StreamId.of(entry.getKey()), entry.getValue());
      }

      Consumer result;
      for (Consumer consumer : consumers.values()) {
        try {
          result = consumer.onReconnect();
        } catch (DrivelineException ignored) {
          result = null;
        }
        handleConsumerResult(consumer, result);
      }
    } catch (Exception e) {
      log.error("failure while reconnecting", e);
    }
  }

  public void onDisconnect() {
    Consumer result;
    for (Consumer consumer : consumers.values()) {
      try {
        result = consumer.onDisconnect();
      } catch (DrivelineException ignored) {
        result = null;
      }
      handleConsumerResult(consumer, result);
    }
  }

  public void onError(String error) {
    log.error("connection failed: {}", error);
  }

  public void onMessage(byte[] message, int offset, int length) {
    try {
      ServerMessage serverMessage = ServerMessage.fromBytes(decoder, message, offset, length);
      Consumer consumer = consumers.get(serverMessage.consumerID);
      if (consumer == null) {
        log.error("received data for an unregistered consumer {}", serverMessage.consumerID);
        return;
      }
      Consumer result = consumer.onMessage(serverMessage);
      handleConsumerResult(consumer, result);
    } catch (Exception e) {
      log.error("error while ", e);
      // TODO: call error handler
      // TODO: add stat for invalid message
    }
  }

  private void handleConsumerResult(Consumer current, Consumer next) {
    if (next == null) {
      try {
        current.close();
      } catch (DrivelineException ignored) {
      }
      unregisterConsumer(current);
    } else if (current != next) {
      registerConsumer(next);
    }
  }

  private int nextConsumerID() {
    return nextConsumerId.getAndIncrement();
  }

  @SuppressWarnings({"unused", "WeakerAccess"})
  public static class Builder {

    private URI endpoint;
    private Transport transport;

    public Builder endpoint(String endpoint) {
      return endpoint(URI.create(endpoint));
    }

    public Builder endpoint(URI endpoint) {
      if (endpoint == null) {
        throw new IllegalArgumentException("endpoint must be specified");
      }
      try {
        String scheme = endpoint.getScheme();
        if (!scheme.equals("ws") && !scheme.equals("wss")) {
          throw new IllegalArgumentException("endpoint must be a WebSocket URI");
        }
      } catch (Exception e) {
        throw new IllegalArgumentException("invalid endpoint");
      }
      this.endpoint = endpoint;
      return this;
    }

    public Builder transport(Transport transport) {
      if (transport == null) {
        throw new IllegalArgumentException("transport must be specified");
      }
      this.transport = transport;
      return this;
    }

    public DrivelineClient build() {
      Transport t = transport != null ? transport : new SyncTransport();
      return new DrivelineClient(endpoint, t);
    }

  }

  public static Builder builder() {
    return new Builder();
  }
}
