// Copyright 2022-2023 Buf Technologies, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package build.buf.connect.extensions

import build.buf.connect.AnyError
import build.buf.connect.CompletionParser
import build.buf.connect.ConnectErrorDetail
import build.buf.connect.generated.v1.Status
import com.squareup.wire.AnyMessage
import com.squareup.wire.Message
import kotlin.reflect.KClass
import kotlin.reflect.full.createInstance
import okio.ByteString.Companion.decodeBase64

private const val TYPE_URL_PREFIX = "type.googleapis.com/"

internal object WireErrorParser : CompletionParser {
    override fun <E : Any> unpack(any: AnyError, clazz: KClass<E>): E? {
        val instance = clazz.createInstance() as Message<*, *>
        val anyMessage = AnyMessage(
            if (any.typeUrl.contains('/')) any.typeUrl else "$TYPE_URL_PREFIX${any.typeUrl}",
            any.value.utf8().decodeBase64() ?: any.value,
        )
        val unpacked = anyMessage.unpackOrNull(instance.adapter)
        if (unpacked?.javaClass?.isAssignableFrom(clazz.java) == true) {
            return unpacked as E?
        }
        return null
    }

    override fun parseDetails(bytes: ByteArray): List<ConnectErrorDetail> {
        val status = Status.ADAPTER.decode(bytes)
        return status.details.map { msg ->
            ConnectErrorDetail(
                msg.typeUrl,
                // Try to decode via base64 and if that fails, use the original value.
                // Connect unary ends up encoding the payload as base64. GRPC and GRPC-Web
                // both do not encode this payload as base64 so decodeBase64() returns null.
                msg.value.utf8().decodeBase64() ?: msg.value
            )
        }
    }
}
