package be.personify.util.io;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * Utility class for IO
 * @author vanderw
 *
 */
public class IOUtils {
	
	
	public static final String CHARSET_UTF_8 = "UTF-8";
	
	/**
	 * Read a Inputstream
	 * @param inputStream containign the things to read
	 * @return a byte array cintaing the things that are read
	 * @throws IOException containing the stacktrace
	 */
	public static byte[] readFileAsBytes(InputStream inputStream) throws IOException {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
	    int nbytes = 0;
	    byte[] buffer = new byte[1024*4];

	    try {
	        while ((nbytes = inputStream.read(buffer)) != -1) {
	            out.write(buffer, 0, nbytes);
	        }
	        return out.toByteArray();
	    } 
	    finally {
	        if (inputStream != null) { 
	        	inputStream.close();
	        }
	        if (out != null) {
	            out.close();
	        }
	    }    
	}
	
	
	/**
	 * Returns the default charset
	 * @return the default charset
	 */
	public static final Charset defaultCharset() {
		return Charset.forName(CHARSET_UTF_8);
	}
	
	
	/**
	 * Reads a file as bytes
	 * @param c the class for the getresourceasstream
	 * @param fileName the name of the file
	 * @return the byte array containing the things read
	 * @throws IOException containing the stacktrace
	 */
	public static byte[] readFileAsBytes(Class<?> c, String fileName) throws IOException {
	    InputStream inputStream = new BufferedInputStream(c.getClassLoader().getResourceAsStream(fileName));
	    return readFileAsBytes(inputStream);
	}
	
	
	/**
	 * Returns the name/ip
	 * @return string containing the hostname/ip of the localhost
	 */
	public static String getIp() {
		try {
			return InetAddress.getLocalHost().toString();
		} catch (UnknownHostException e) {
			e.printStackTrace();
		}
		return "unknown";
	}
	
	
	
	public static void zip(String sourceDirPath, String zipFilePath) throws IOException {
	    Path p = Files.createFile(Paths.get(zipFilePath));
	    Path pp = Paths.get(sourceDirPath);
	    try (ZipOutputStream zs = new ZipOutputStream(Files.newOutputStream(p));
	        Stream<Path> paths = Files.walk(pp)) {
	        paths
	          .filter(path -> !Files.isDirectory(path))
	          .forEach(path -> {
	              ZipEntry zipEntry = new ZipEntry(pp.relativize(path).toString());
	              try {
	                  zs.putNextEntry(zipEntry);
	                  Files.copy(path, zs);
	                  zs.closeEntry();
	            } catch (IOException e) {
	                System.err.println(e);
	            }
	          });
	    }
	}
	

}
