package be.personify.audit.kafka;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;

import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.serialization.StringSerializer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.fasterxml.jackson.databind.ObjectMapper;

import be.personify.audit.AuditLoggerImplementations;
import be.personify.util.Constants;
import be.personify.util.StringUtils;
import be.personify.util.audit.AuditLogger;
import be.personify.util.audit.InvalidConfigurationException; 

/**
 * audit logging implementation for kafka
 * 
 * @author wvdb
 *
 */ 
public class AuditLoggerKafkaImpl implements AuditLogger {
	
	private static final Logger logger = LogManager.getLogger(AuditLoggerKafkaImpl.class.getName());
	
	private static final String OBJECT = "object";
	private static final String ACTION = "action";
	private static final String ENTITY = "entity";

	private KafkaProducer<String, Object> producer = null;
	

	private static final String PROPERTY_TOPIC = "topic";
	
	private String topic = null;
	
	private ObjectMapper objectMapper = Constants.objectMapper;
 
	@Override
	public void configure(Map<String, Object> configuration) throws InvalidConfigurationException {
		
		 String bootstrapservers = (String)configuration.get(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG);
		 if ( StringUtils.isEmpty(bootstrapservers)) {
			 logger.error("property {} not found ", ProducerConfig.BOOTSTRAP_SERVERS_CONFIG);
			 throw new InvalidConfigurationException("property not found : " + ProducerConfig.BOOTSTRAP_SERVERS_CONFIG);
		 }
		    
		 topic = (String)configuration.get(PROPERTY_TOPIC);
		 if ( StringUtils.isEmpty(topic)) {
		  	throw new InvalidConfigurationException("topic property not found : " + PROPERTY_TOPIC);
		 }
		
		 Properties properties = new Properties();
		 properties.setProperty(ProducerConfig.KEY_SERIALIZER_CLASS_CONFIG, StringSerializer.class.getName());
		 properties.setProperty(ProducerConfig.VALUE_SERIALIZER_CLASS_CONFIG, StringSerializer.class.getName());
	    
		 for ( Entry<String,Object> entry : configuration.entrySet()) {
			 properties.setProperty(entry.getKey(), (String)entry.getValue());
		 }
	    
	   
	    
		
	    producer = new KafkaProducer<>(properties);
		
	}

	@Override
	public void log(String entity, String action, String id, Object object) {
		
		Map<String,Object> m = new HashMap<String, Object>();
		m.put(ENTITY, entity);
		m.put(ACTION, action);
		m.put(OBJECT, object);
		
		String mapString = m.toString();
		try {
			mapString = objectMapper.writeValueAsString(m);
		}
		catch ( Exception e ) {
			e.printStackTrace();
		}
		
		ProducerRecord<String, Object> producerRecord =  new ProducerRecord<String,Object>(topic, mapString );
	    producer.send(producerRecord);
        producer.flush();
	}

	@Override
	public String name() {
		return AuditLoggerImplementations.kafka.name();
	}

}
