package ws.osiris.awsdeploy

import com.amazonaws.services.apigateway.AmazonApiGatewayClientBuilder
import com.amazonaws.services.apigateway.model.CreateDeploymentRequest
import com.amazonaws.services.s3.AmazonS3ClientBuilder
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory
import org.slf4j.LoggerFactory
import ws.osiris.aws.Stage
import java.nio.file.Path

private val log = LoggerFactory.getLogger("ws.osiris.awsdeploy")

/**
 * Deploys the API to the stages and returns the names of the stages that were updated.
 *
 * If the API is being deployed for the first time then all stages are deployed. If the API
 * was updated then only stages where `deployOnUpdate` is true are deployed.
 */
fun deployStages(
    profile: AwsProfile,
    apiId: String,
    apiName: String,
    stages: List<Stage>,
    stackCreated: Boolean
): List<String> {
    // no need to deploy stages if the stack has just been created
    return if (stackCreated) {
        stages.map { it.name }
    } else {
        val apiGateway = AmazonApiGatewayClientBuilder.standard()
            .withCredentials(profile.credentialsProvider)
            .withRegion(profile.region)
            .build()
        val stagesToDeploy = stages.filter { it.deployOnUpdate }
        for (stage in stagesToDeploy) {
            log.debug("Updating REST API '$apiName' in stage '${stage.name}'")
            apiGateway.createDeployment(CreateDeploymentRequest().apply {
                restApiId = apiId
                stageName = stage.name
                variables = stage.variables
                description = stage.description
            })
        }
        stagesToDeploy.map { it.name }
    }
}

/**
 * Creates an S3 bucket to hold static files.
 *
 * The bucket name is `${API name}.static-files`, converted to lower case.
 *
 * If the bucket already exists the function does nothing.
 */
fun createBucket(
    profile: AwsProfile,
    apiName: String,
    envName: String?,
    suffix: String,
    prefix: String?
): String {
    val s3Client = AmazonS3ClientBuilder.standard()
        .withCredentials(profile.credentialsProvider)
        .withRegion(profile.region)
        .build()
    val bucketName = bucketName(apiName, envName, suffix, prefix)
    if (!s3Client.doesBucketExistV2(bucketName)) {
        s3Client.createBucket(bucketName)
        log.info("Created S3 bucket '$bucketName'")
    } else {
        log.info("Using existing S3 bucket '$bucketName'")
    }
    return bucketName
}

/**
 * Uploads a file to an S3 bucket and returns the URL of the file in S3.
 */
fun uploadFile(profile: AwsProfile, file: Path, bucketName: String, key: String? = null): String =
    uploadFile(profile, file, bucketName, file.parent, key)

/**
 * Uploads a file to an S3 bucket and returns the URL of the file in S3.
 *
 * The file should be under `baseDir` on the filesystem. The S3 key for the file will be the relative path
 * from the base directory to the file.
 *
 * For example, if `baseDir` is `/foo/bar` and the file is `/foo/bar/baz/qux.txt` then the file will be
 * uploaded to S3 with the key `baz/qux.txt
 *
 * The key can be specified by the caller in which case it is used instead of automatically generating
 * a key.
 */
fun uploadFile(
    profile: AwsProfile,
    file: Path,
    bucketName: String,
    baseDir: Path,
    key: String? = null,
    bucketDir: String? = null
): String {
    val s3Client = AmazonS3ClientBuilder.standard()
        .withCredentials(profile.credentialsProvider)
        .withRegion(profile.region)
        .build()
    val uploadKey = key ?: baseDir.relativize(file).toString()
    val dirPart = bucketDir?.let { "$bucketDir/" } ?: ""
    val fullKey = "$dirPart$uploadKey"
    s3Client.putObject(bucketName, fullKey, file.toFile())
    val url = "https://s3-${profile.region}.amazonaws.com/$bucketName/$fullKey"
    log.debug("Uploaded file {} to S3 bucket {}, URL {}", file, bucketName, url)
    return url
}

/**
 * Returns the name of a bucket for the group and API with the specified suffix.
 *
 * The bucket name is `<API name>.<suffix>`
 */
fun bucketName(apiName: String, envName: String?, suffix: String, prefix: String?): String {
    val accountPart = if (envName == null) "" else "$envName."
    val prefixPart = if (prefix == null) "" else "$prefix."
    return "$prefixPart$apiName.$accountPart$suffix"
}

/**
 * Returns the default name of the S3 bucket from which code is deployed
 */
fun codeBucketName(apiName: String, envName: String?, prefix: String?): String =
    bucketName(apiName, envName, "code", prefix)

/**
 * Returns the name of the static files bucket for the API.
 */
fun staticFilesBucketName(apiName: String, envName: String?, prefix: String?): String =
    bucketName(apiName, envName, "static-files", prefix)

/**
 * Equivalent of Maven's `MojoFailureException` - indicates something has failed during the deployment.
 */
class DeployException(msg: String) : RuntimeException(msg)

/**
 * Parses `root.template` and returns a set of all parameter names passed to the generated CloudFormation template.
 *
 * These are passed to the lambda as environment variables. This allows the handler code to refer to any
 * AWS resources defined in `root.template`.
 *
 * This allows (for example) for lambda functions to be defined in the project, created in `root.template`
 * and referenced in the project via environment variables.
 */
@Suppress("UNCHECKED_CAST")
internal fun generatedTemplateParameters(templateYaml: String, codeBucketName: String, apiName: String): Set<String> {
    val objectMapper = ObjectMapper(YAMLFactory())
    val rootTemplateMap = objectMapper.readValue(templateYaml, Map::class.java)
    val generatedTemplateUrl = "https://s3-\${AWS::Region}.amazonaws.com/\${bucketPrefix}$codeBucketName/$apiName.template"
    val parameters = (rootTemplateMap["Resources"] as Map<String, Any>?)
        ?.map { it.value as Map<String, Any> }
        ?.filter { it["Type"] == "AWS::CloudFormation::Stack" }
        ?.map { it["Properties"] as Map<String, Any> }
        ?.filter { it["TemplateURL"] == generatedTemplateUrl }
        ?.map { it["Parameters"] as Map<String, String> }
        ?.map { it.keys }
        ?.singleOrNull() ?: setOf()
    // The LambdaRole parameter is used by Osiris and doesn't need to be passed to the user code
    return parameters - "LambdaRole"
}

