/*
 * MIT License
 *
 * Copyright (c) 2018 MODL
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package uk.num.json_modl.converter;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.function.Function;
import java.util.regex.Pattern;

class UtilFuncs {
    /**
     * Escape graves.
     */
    static final Function<String, String> escapeGraves = (s) -> (s != null) ? StringUtils.replace(s, "`", "~`") : s;

    private static final Pattern shouldBeGraveQuoted = Pattern.compile(".*[()\\[\\];{}:=].*");
    private static final ObjectMapper mapper = new ObjectMapper();
    /**
     * Some MODL string need backtick quotes.
     */
    private static final Function<String, String> graveQuoteIfNecessary = (s) -> s != null && ((shouldBeGraveQuoted.matcher(s)
            .matches()) || StringUtils.isEmpty(s)) ? "`" + s + "`" : s;
    /**
     * Some MODL string need backtick quotes.
     */
    private static final Function<String, String> doubleQuoteIfNecessary = (s) -> s != null && (s.contains("~`")) ? "\"" + s + "\"" : s;
    /**
     * Handy function composition
     */
    static final Function<String, String> escapeAndQuote = doubleQuoteIfNecessary.compose(UtilFuncs.graveQuoteIfNecessary)
            .compose(escapeGraves)
            .compose(UnicodeEscaper.escape);
    /**
     * Convert JSON to a JsonNode
     */
    static Function<String, JsonNode> mapJson = (s) -> {
        try {
            return (mapper.readTree(s));
        } catch (final IOException e) {
            throw new RuntimeException(e);
        }
    };
    /**
     * Convert an ArrayNode to a List of JsonNodes
     */
    static Function<ArrayNode, List<JsonNode>> toList = (node) -> {
        final List<JsonNode> nodes = new ArrayList<>();

        for (int i = 0; i < node.size(); i++) {
            nodes.add(node.get(i));
        }
        return nodes;
    };
    /**
     * Convert the elements of a JsonNode to a List of NamedNode objects
     */
    static Function<JsonNode, List<NamedNode>> toNamedNodeList = (node) -> {
        final Iterator<String> elements = node.fieldNames();
        final List<NamedNode> nodes = new ArrayList<>();

        while (elements.hasNext()) {
            final String field = elements.next();
            nodes.add(new NamedNode(field, node.get(field)));
        }
        return nodes;
    };

}
