/*
 * MIT License
 *
 * Copyright (c) 2018 MODL
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package uk.num.json_modl.converter;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.*;
import lombok.extern.log4j.Log4j2;

import java.util.List;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * A class to convert a JSON String to a MODL String.
 *
 * @author tonywalmsley
 */
@Log4j2
public class JsonToModl {

    /**
     * Convert a JSON primitive node to a primitive MODL value
     */
    private static final Function<JsonNode, Optional<String>> primitiveNodeToModl = (node) -> {
        //
        // Handling for nulls
        //
        if (node instanceof NullNode) {
            return Optional.of("000");
        }
        if (node instanceof TextNode) {
            if ("null".equals(node.textValue())) {
                return Optional.of("`null`");
            } else {
                return Optional.of(UtilFuncs.escapeAndQuote.apply(node.textValue()));
            }
        }
        //
        // Handling for booleans
        //
        if (node instanceof BooleanNode) {
            final boolean bool = node.booleanValue();
            final String boolString = (bool) ? "01" : "00";
            return Optional.of(boolString);
        }
        //
        // Handling for numbers
        //
        if (node instanceof NumericNode) {
            return Optional.of("" + node.numberValue());
        }
        return Optional.empty();
    };

    /**
     * Convert a JSON primitive node to a primitive MODL value
     */
    private static final BiFunction<String, JsonNode, Optional<String>> primitiveToModl = (escapedKey, node) -> {
        final Optional<String> nodeAsString = primitiveNodeToModl.apply(node);
        return nodeAsString.map(s -> escapedKey + "=" + s);
    };

    /**
     * Render a JSON Object as a MODL Map but without the surrounding parentheses
     */
    private static final Function<ObjectNode, String> objectToModlMapWithoutParentheses = (node) -> UtilFuncs.toNamedNodeList.apply(node)
            .stream()
            .map(namedNode -> pairToModl(namedNode.name, namedNode.node))
            .collect(Collectors.joining(";"));

    /**
     * Convert the `elements` in a JsonNode to MODL
     */
    private static final Function<JsonNode, String> toModl = (node) -> {
        if (node instanceof ArrayNode) {
            return arrayToModlArray((ArrayNode) node);
        }
        if (node instanceof ObjectNode) {
            return "(" + objectToModlMapWithoutParentheses.apply((ObjectNode) node) + ")";
        }
        if (node.elements()
                .hasNext()) {
            final List<NamedNode> nodes = UtilFuncs.toNamedNodeList.apply(node);

            return nodes
                    .stream()
                    .map(entry -> pairToModl(entry.name, entry.node))
                    .collect(Collectors.joining(";"));
        }
        return primitiveNodeToModl.apply(node)
                .orElse("");
    };

    /**
     * Convert the `elements` in a JsonNode to MODL
     */
    private static final Function<JsonNode, String> toTopLevelModl = (node) -> {
        if (node instanceof ObjectNode) {
            return objectToModlMapWithoutParentheses.apply((ObjectNode) node);
        }
        return toModl.apply(node);
    };

    /**
     * Convert a named JsonNode to MODL
     *
     * @param key  the node name String
     * @param node the JsonNode
     * @return a MODL String
     */
    private static String pairToModl(final String key, final JsonNode node) {
        final String escapedKey = UnicodeEscaper.escape.apply(key);
        if (node instanceof ArrayNode) {
            return escapedKey + arrayToModlArray((ArrayNode) node);
        }
        if (node instanceof ObjectNode) {
            return escapedKey + "(" + objectToModlMapWithoutParentheses.apply((ObjectNode) node) + ")";
        }
        return primitiveToModl.apply(escapedKey, node)
                .orElseGet(() -> escapedKey + "(" + toModl.apply(node) + ")");
    }

    /**
     * Render a JSON array as a MODL Array
     *
     * @param node the ArrayNode
     * @return the MODL String
     */
    private static String arrayToModlArray(final ArrayNode node) {
        return "[" + UtilFuncs.toList.apply(node)
                .stream()
                .map(toModl)
                .collect(Collectors.joining(";")) + "]";
    }

    /**
     * Convert a JSON String to a MODL String
     *
     * @param json the JSON String
     * @return the MODL String
     */
    public String pairToModl(final String json) {
        log.debug("Converting: {}", json);
        final JsonNode node = UtilFuncs.mapJson.apply(json);

        //
        // Top-level items need special handling
        //
        final String modlResult;
        if (node instanceof ArrayNode) {
            modlResult = pairToModl("", node);
        } else {
            modlResult = toTopLevelModl.apply(node);
        }

        log.debug("Result    : {}", modlResult);

        return modlResult;
    }

}
