package uk.num.json_modl.converter;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.*;
import lombok.extern.log4j.Log4j2;

import java.util.List;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * A class to convert a JSON String to a MODL String.
 *
 * @author tonywalmsley
 */
@Log4j2
public class JsonToModl {
    /**
     * Convert a JSON primitive node to a primitive MODL value
     */
    private static BiFunction<String, JsonNode, Optional<String>> primitiveToModl = (escapedKey, node) -> {
        //
        // Handling for nulls
        //
        if (node instanceof NullNode) {
            return Optional.of(escapedKey + "=000");
        }
        if (node instanceof TextNode) {
            if ("null".equals(node.textValue())) {
                return Optional.of(escapedKey + "=`null`");
            } else {
                return Optional.of(escapedKey + "=" + UtilFuncs.escapeAndQuote.apply(node.textValue()));
            }
        }
        //
        // Handling for booleans
        //
        if (node instanceof BooleanNode) {
            final boolean bool = node.booleanValue();
            final String boolString = (bool) ? "01" : "00";
            return Optional.of(escapedKey + "=" + boolString);
        }
        //
        // Handling for numbers
        //
        if (node instanceof NumericNode) {
            return Optional.of(escapedKey + "=" + node.numberValue());
        }
        return Optional.empty();
    };
    /**
     * Render a JSON Object as a MODL Map but without the surrounding parentheses
     */
    private static Function<ObjectNode, String> objectToModlMapWithoutParentheses = (node) -> UtilFuncs.toNamedNodeList.apply(node)
            .stream()
            .map(namedNode -> pairToModl(namedNode.name, namedNode.node))
            .collect(Collectors.joining(";"));
    /**
     * Convert the `elements` in a JsonNode to MODL
     */
    private static Function<JsonNode, String> toModl = (node) -> {
        if (node instanceof ArrayNode) {
            return arrayToModlArray((ArrayNode) node);
        }
        if (node instanceof ObjectNode) {
            return "(" + objectToModlMapWithoutParentheses.apply((ObjectNode) node) + ")";
        }
        if (node.elements()
                .hasNext()) {
            final List<NamedNode> nodes = UtilFuncs.toNamedNodeList.apply(node);

            return nodes
                    .stream()
                    .map(entry -> pairToModl(entry.name, entry.node))
                    .collect(Collectors.joining(";"));
        }
        return node.asText();
    };
    /**
     * Convert the `elements` in a JsonNode to MODL
     */
    private static Function<JsonNode, String> toTopLevelModl = (node) -> {
        if (node instanceof ObjectNode) {
            return objectToModlMapWithoutParentheses.apply((ObjectNode) node);
        }
        return toModl.apply(node);
    };

    /**
     * Convert a named JsonNode to MODL
     *
     * @param key  the node name String
     * @param node the JsonNode
     * @return a MODL String
     */
    private static String pairToModl(final String key, final JsonNode node) {
        final String escapedKey = UnicodeEscaper.escape.apply(key);
        if (node instanceof ArrayNode) {
            return escapedKey + arrayToModlArray((ArrayNode) node);
        }
        if (node instanceof ObjectNode) {
            return escapedKey + "(" + objectToModlMapWithoutParentheses.apply((ObjectNode) node) + ")";
        }
        return primitiveToModl.apply(escapedKey, node)
                .orElseGet(() -> escapedKey + "(" + toModl.apply(node) + ")");
    }

    /**
     * Render a JSON array as a MODL Array
     *
     * @param node the ArrayNode
     * @return the MODL String
     */
    private static String arrayToModlArray(final ArrayNode node) {
        return "[" + UtilFuncs.toList.apply(node)
                .stream()
                .map(toModl)
                .collect(Collectors.joining(";")) + "]";
    }

    /**
     * Convert a JSON String to a MODL String
     *
     * @param json the JSON String
     * @return the MODL String
     */
    public String pairToModl(final String json) {
        log.info("Converting: {}", json);
        final JsonNode node = UtilFuncs.mapJson.apply(json);

        //
        // Top-level items need special handling
        //
        final String modlResult;
        if (node instanceof ArrayNode) {
            modlResult = pairToModl("", node);
        } else {
            modlResult = toTopLevelModl.apply(node);
        }

        //
        // Check the generated MODL
        //
        final String result = ModlVerifier.verifyModl.apply(modlResult, json);

        log.info("Result    : {}", result);

        return result;
    }
}
