/*
 * This file is part of AuthKit.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * Copyright (C) hdsdi3g for hd3g.tv 2019
 *
 */
package tv.hd3g.authkit.mod.service;

import static jakarta.transaction.Transactional.TxType.REQUIRES_NEW;

import java.util.Arrays;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import jakarta.transaction.Transactional;
import tv.hd3g.authkit.mod.controller.RestControllerUser;
import tv.hd3g.authkit.mod.dto.Password;
import tv.hd3g.authkit.mod.dto.validated.AddGroupOrRoleDto;
import tv.hd3g.authkit.mod.dto.validated.AddUserDto;
import tv.hd3g.authkit.mod.exception.BlockedUserException;
import tv.hd3g.authkit.mod.exception.ResetWithSamePasswordException;
import tv.hd3g.authkit.mod.repository.CredentialRepository;
import tv.hd3g.commons.authkit.CheckBefore;

@Service
public class CmdLineServiceImpl implements CmdLineService {
	private static Logger log = LogManager.getLogger();

	@Autowired
	private CredentialRepository credentialRepository;
	@Autowired
	private AuthenticationService authenticationService;

	@Value("${authkit.realm:default}")
	private String realm;

	/**
	 * REQUIRES_NEW: workaround for this problem https://stackoverflow.com/questions/24338150/how-to-manually-force-a-commit-in-a-transactional-method
	 */
	@Override
	@Transactional(REQUIRES_NEW)
	public void addOrUpdateSecurityAdminUser(final String login,
											 final Password password) throws ResetWithSamePasswordException {
		final var previousUser = credentialRepository.getFromRealmLogin(realm, login);
		final String userUUID;
		if (previousUser != null) {
			userUUID = previousUser.getUser().getUuid();
			authenticationService.resetUserLogonTrials(userUUID);
			try {
				authenticationService.changeUserPassword(previousUser.getUser().getUuid(), password);
			} catch (final BlockedUserException e) {
				log.error("Unexpected error", e);
			}
			authenticationService.enableUser(userUUID);
		} else {
			final var addUserDto = new AddUserDto();
			addUserDto.setUserLogin(login);
			addUserDto.setUserPassword(password);
			userUUID = authenticationService.addUser(addUserDto);
		}

		final var g = new AddGroupOrRoleDto();
		g.setName("SecurityAdmins");
		g.setDescription("Autogenerated group for " + login);
		authenticationService.addGroup(g);
		authenticationService.addUserInGroup(userUUID, g.getName());

		final var r = new AddGroupOrRoleDto();
		r.setName("SecurityOnly");
		r.setName("Autogenerated role for " + login);
		authenticationService.addRole(r);

		authenticationService.addGroupInRole(g.getName(), r.getName());

		Arrays.stream(RestControllerUser.class.getAnnotationsByType(CheckBefore.class)).findFirst().map(
				cb -> Arrays.stream(cb.value())).ifPresent(
						cb -> cb.forEach(
								rightName -> authenticationService.addRightInRole(r.getName(), rightName)));
	}

}
