/*
 * Copyright 2015 Ryan Svihla
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pro.foundev.cassandra.test.mapper;

import com.datastax.driver.core.ResultSet;
import com.datastax.driver.mapping.Mapper;
import com.datastax.driver.mapping.MappingManager;
import com.datastax.driver.mapping.annotations.Table;
import pro.foundev.cassandra.commons.core.CassandraSessionFactory;
import pro.foundev.cassandra.commons.test.CassandraTestDB;
import pro.foundev.cassandra.commons.test.FullTableName;

public class CassandraMapperTestDB extends CassandraTestDB {

    private final MappingManager mappingManager;
    public CassandraMapperTestDB(CassandraSessionFactory cassandraSessionFactory){
        super(cassandraSessionFactory);
        mappingManager = new MappingManager(session);
    }

       /**
     * Upserts a record to the database. There is no validation of previously existing
     * @param klazz
     * @param t
     * @param <T>
     */
    public <T> void save(Class<T> klazz, T t) {
        getOrCreateMapper(klazz).save(t);
        markForCleanUp(klazz);
    }


    /**
     * Gets the count of a given table. Not on v1 protocol servers such as pre Cassandra 2.0 the default limit of 5k
     * will be in place. For 2.0 and later paging will take over on larger queries and there should be no limit.
     * @param klazz class needed for interrogation of @Table annotation and it depends on the optional
     *              keyspace parameter being set today
     * @return Total count of the table
     */
    public Long count(Class<?> klazz) {
        FullTableName fullTableName = getTableNameFromClass(klazz);
        ResultSet resultSet = session.execute("SELECT COUNT(*) FROM " + fullTableName);
        Long count = resultSet.one().getLong(0);
        return count;
    }


    /**
     * simple method to make a table for cleanup in case you've not managed to visit it by other means.
     * This should not be the preferred method of marking for cleanup, and one should rely on the get, delete,
     * and save functions.
     * @param klazz must have @Table annotation from datastax mapping project
     */
    public synchronized void markForCleanUp(Class<?> klazz) {
        FullTableName fullTableName = getTableNameFromClass(klazz);
        super.markForCleanUp(fullTableName);
    }

     /**
     * Creates a mapper based on the class passed in. The class will need the @Table annotation from the
     * DataStax driver.
     * FIXME: retain mappers in a cache
     * @param klazz
     * @param <T>
     * @return returns a Mapper
     */
    private <T> Mapper<T> getOrCreateMapper(Class<T> klazz){
        Mapper<T> mapper = mappingManager.mapper(klazz);
        return mapper;
    }

    /**
     * FIXME: retain table class name cache
     * @param klazz class needed for interrogation of @Table annotation. If there is a logged keyspace attached
     *              to the session it will use that otherwise, it will depends on the optional keyspace parameter
     *              of the table annotation
     * @return Fully qualified string of keyspace.tablename
     */
    private FullTableName getTableNameFromClass(Class<?> klazz){
        Table tableAnnotation = klazz.getAnnotation(Table.class);
        String keyspace;
        String loggedKeyspace = session.getLoggedKeyspace();
        if(loggedKeyspace!=null){
            keyspace = loggedKeyspace;
        }else {
            keyspace = tableAnnotation.keyspace();
        }
        String name = tableAnnotation.name();
        FullTableName fullTableName = new FullTableName();
        fullTableName.setKeyspaceName(keyspace);
        fullTableName.setTableName(name);
        return fullTableName;
    }


    /**
     * Gets a record by full primary key and marks the table for cleanup later.
     * @param klazz
     * @param keys must contain all objects that make up the primary key
     * @param <T> Must be a class with the DataStax Java Driver @Table annotation
     * @return returns a single instance of the specified class from Cassandra
     */
    public <T> T get(Class<T> klazz,  Object ... keys) {
        return getOrCreateMapper(klazz).get(keys);
    }

    /**
     *  Deletes a record. Does not handle cleanup
     * @param klazz
     * @param keys must contain all objects that make up the primary key
     * @param <T> Must be a class with the DataStax Java Driver @Table annotation
     */
    public <T> void delete(Class<T> klazz, Object ... keys){
        getOrCreateMapper(klazz).delete(keys);
    }

}
