/*
 *    Copyright 2009-2023 the original author or authors.
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *       https://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package pro.chenggang.project.reactive.mybatis.support.generator.support;

import org.apache.commons.lang3.StringUtils;
import org.mybatis.generator.api.IntrospectedColumn;
import org.mybatis.generator.api.IntrospectedTable;
import org.mybatis.generator.api.dom.java.Field;
import org.mybatis.generator.api.dom.java.FullyQualifiedJavaType;
import org.mybatis.generator.api.dom.java.Interface;
import org.mybatis.generator.api.dom.java.JavaVisibility;
import org.mybatis.generator.api.dom.java.TopLevelClass;
import org.mybatis.generator.config.Context;
import pro.chenggang.project.reactive.mybatis.support.generator.properties.GeneratorProperties;
import pro.chenggang.project.reactive.mybatis.support.generator.properties.GeneratorPropertiesHolder;

import java.util.Objects;
import java.util.Set;

/**
 * Generated model customizer
 *
 * @author Gang Cheng
 * @version 1.0.0
 * @since 1.0.0
 */
public class GeneratedModelCustomizer {

    /**
     * Customize lombok generated if configured.
     *
     * @param interfaze         the top level class
     * @param introspectedTable the introspected table
     */
    public void customizeLombokGeneratedIfConfigured(Interface interfaze, IntrospectedTable introspectedTable){
        GeneratorProperties generatorProperties = GeneratorPropertiesHolder.getInstance().getGeneratorProperties();
        Set<String> lombokAnnotations = generatorProperties.getLombokAnnotations();
        if(!lombokAnnotations.contains("lombok.Generated")){
            return;
        }
        interfaze.addImportedType(new FullyQualifiedJavaType("lombok.Generated"));
        interfaze.addAnnotation("@Generated");
    }

    /**
     * customize lombok
     *
     * @param topLevelClass     the TopLevelClass
     * @param introspectedTable the IntrospectedTable
     */
    public void customizeLombok(TopLevelClass topLevelClass, IntrospectedTable introspectedTable) {
        String tableRemarks = StringUtils.defaultIfBlank(introspectedTable.getRemarks(), "auto generated");
        topLevelClass.addJavaDocLine("/**");
        topLevelClass.addJavaDocLine(" * " + tableRemarks);
        topLevelClass.addJavaDocLine(" * ");
        topLevelClass.addJavaDocLine(" * @author AutoGenerated");
        topLevelClass.addJavaDocLine(" */");
        GeneratorProperties generatorProperties = GeneratorPropertiesHolder.getInstance().getGeneratorProperties();
        topLevelClass.getFields().forEach(field -> {
            field.setVisibility(generatorProperties.getFieldVisibility());
            field.setFinal(generatorProperties.isFinalField());
        });
        Set<String> lombokConfigs = generatorProperties.getLombokAnnotations();
        if (Objects.isNull(lombokConfigs) || lombokConfigs.isEmpty()) {
            return;
        }
        lombokConfigs.forEach(lombokAnnotationName -> {
            if(!StringUtils.startsWith(lombokAnnotationName,"lombok")){
                return;
            }
            String lombokImport;
            String lombokAnnotation;
            if(lombokAnnotationName.contains("(")){
                String pureAnnotation = StringUtils.substringBefore(StringUtils.substringAfterLast(StringUtils.substringBefore(lombokAnnotationName,"("),"."),  "(");
                lombokImport = StringUtils.substringBefore(lombokAnnotationName,"(");
                lombokAnnotation = StringUtils.substringAfter(lombokAnnotationName,
                        StringUtils.substringBeforeLast(lombokImport, pureAnnotation)
                );
            }else {
                lombokAnnotation = StringUtils.substringAfterLast(lombokAnnotationName,".");
                lombokImport = lombokAnnotationName;
            }
            topLevelClass.addImportedType(lombokImport);
            topLevelClass.addAnnotation("@" + lombokAnnotation);
        });
    }

    /**
     * customize model constant
     *
     * @param topLevelClass     the TopLevelClass
     * @param introspectedTable the IntrospectedTable
     */
    public void customizeModelConstant(Context context, TopLevelClass topLevelClass, IntrospectedTable introspectedTable) {
        for (IntrospectedColumn introspectedColumn : introspectedTable.getAllColumns()) {
            Field field = new Field(introspectedColumn.getActualColumnName().toUpperCase(), new FullyQualifiedJavaType(String.class.getName()));
            field.setVisibility(JavaVisibility.PUBLIC);
            field.setStatic(true);
            field.setFinal(true);
            field.setInitializationString("\"" + introspectedColumn.getJavaProperty() + "\"");
            context.getCommentGenerator().addClassComment(topLevelClass, introspectedTable);
            topLevelClass.addField(field);
            Field columnField = new Field("DB_" + introspectedColumn.getActualColumnName().toUpperCase(), new FullyQualifiedJavaType(String.class.getName()));
            columnField.setVisibility(JavaVisibility.PUBLIC);
            columnField.setStatic(true);
            columnField.setFinal(true);
            columnField.setInitializationString("\"" + introspectedColumn.getActualColumnName() + "\"");
            topLevelClass.addField(columnField);
        }
    }
}
