/*
 * Copyright (C) 2014 Dmitry Kotlyarov, Dmitriy Rogozhin.
 * All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package apphub;

import apphub.storage.Storage;
import apphub.util.NotFoundException;

import java.io.Serializable;
import java.util.HashMap;

/**
 * @author Dmitry Kotlyarov
 * @since 1.0
 */
public final class Application implements Serializable {
    private static final long serialVersionUID = 1;

    private final Environment environment;
    private final String id;
    private final String url;
    private final HashMap<String, Property> propertiesByIds;
    private final Property[] properties;
    private final HashMap<String, Instance> instancesByIds;
    private final HashMap<String, Instance> instancesByPublicIps;
    private final HashMap<String, Instance> instancesByPrivateIps;
    private final HashMap<String, Instance> instancesByPublicHostnames;
    private final HashMap<String, Instance> instancesByPrivateHostnames;
    private final Instance[] instances;
    private final Storage storage;

    Application(Environment environment, String id, String url, HashMap<String, Property> propertiesByIds, Property[] properties, HashMap<String, Instance> instancesByIds, HashMap<String, Instance> instancesByPublicIps, HashMap<String, Instance> instancesByPrivateIps, HashMap<String, Instance> instancesByPublicHostnames, HashMap<String, Instance> instancesByPrivateHostnames, Instance[] instances, Storage storage) {
        this.environment = environment;
        this.id = id;
        this.url = url;
        this.propertiesByIds = propertiesByIds;
        this.properties = properties;
        this.instancesByIds = instancesByIds;
        this.instancesByPublicIps = instancesByPublicIps;
        this.instancesByPrivateIps = instancesByPrivateIps;
        this.instancesByPublicHostnames = instancesByPublicHostnames;
        this.instancesByPrivateHostnames = instancesByPrivateHostnames;
        this.instances = instances;
        this.storage = storage;
    }

    public Environment getEnvironment() {
        return environment;
    }

    public String getId() {
        return id;
    }

    public String getUrl() {
        return url;
    }

    @SuppressWarnings({"unchecked"})
    public <V> Property<V> findProperty(Class<V> type, String id) {
        Property prop = propertiesByIds.get(id);
        if ((prop != null) && (prop.getType().equals(type))) {
            return prop;
        } else {
            return null;
        }
    }

    public <V> Property<V> getProperty(Class<V> type, String id) {
        Property<V> prop = findProperty(type, id);
        if (prop != null) {
            return prop;
        } else {
            throw new NotFoundException(String.format("Property of type '%s' with id '%s' is not found", type, id));
        }
    }

    public Property[] getProperties() {
        return properties.clone();
    }

    public Instance getInstance() {
        return getInstance(Config.get().getInstance());
    }

    public Instance findInstance(String id) {
        return instancesByIds.get(id);
    }

    public Instance getInstance(String id) {
        Instance inst = findInstance(id);
        if (inst != null) {
            return inst;
        } else {
            throw new NotFoundException(String.format("Instance with id '%s' is not found", id));
        }
    }

    public Instance findInstanceByPublicIp(String ip) {
        return instancesByPublicIps.get(ip);
    }

    public Instance getInstanceByPublicIp(String ip) {
        Instance inst = findInstanceByPublicIp(ip);
        if (inst != null) {
            return inst;
        } else {
            throw new NotFoundException(String.format("Instance with public ip '%s' is not found", ip));
        }
    }

    public Instance findInstanceByPrivateIp(String ip) {
        return instancesByPrivateIps.get(ip);
    }

    public Instance getInstanceByPrivateIp(String ip) {
        Instance inst = findInstanceByPrivateIp(ip);
        if (inst != null) {
            return inst;
        } else {
            throw new NotFoundException(String.format("Instance with private ip '%s' is not found", ip));
        }
    }

    public Instance findInstanceByPublicHostname(String hostname) {
        return instancesByPublicHostnames.get(hostname);
    }

    public Instance getInstanceByPublicHostname(String hostname) {
        Instance inst = findInstanceByPublicHostname(hostname);
        if (inst != null) {
            return inst;
        } else {
            throw new NotFoundException(String.format("Instance with public hostname '%s' is not found", hostname));
        }
    }

    public Instance findInstanceByPrivateHostname(String hostname) {
        return instancesByPrivateHostnames.get(hostname);
    }

    public Instance getInstanceByPrivateHostname(String hostname) {
        Instance inst = findInstanceByPrivateHostname(hostname);
        if (inst != null) {
            return inst;
        } else {
            throw new NotFoundException(String.format("Instance with private hostname '%s' is not found", hostname));
        }
    }

    public Instance[] getInstances() {
        return instances.clone();
    }

    public Storage getStorage() {
        return storage;
    }
}
