package pl.poznan.put.constant;

import org.apache.commons.math3.util.FastMath;

import java.awt.*;
import java.util.Arrays;

/** A set of utility functions to work with palettes and color interpolation. */
public final class Colors {
  // @formatter:off
  /* 33 colors generated by: http://phrogz.net/css/distinct-colors.html
   *
   * As Hex Values
   * #b21800, #cc7033, #b8e600, #3df23d, #ace6d2, #80b3ff, #a280ff, #661a4d,
   * #33000e, #b26559, #594f43, #5c6633, #008c13, #00c2f2, #000033, #7a00e6,
   * #bf3069, #4c1400, #4c3300, #b6bf8f, #80ffb2, #335c66, #303040, #ff40f2,
   * #f20041, #bf9c8f, #d9b836, #17330d, #165943, #1d4b73, #341d73, #cc99c2,
   * #66001b
   *
   * As RGB Values
   * rgb(178,24,0), rgb(204,112,51), rgb(184,230,0), rgb(61,242,61),
   * rgb(172,230,210), rgb(128,179,255), rgb(162,128,255), rgb(102,26,77),
   * rgb(51,0,14), rgb(178,101,89), rgb(89,79,67), rgb(92,102,51),
   * rgb(0,140,19), rgb(0,194,242), rgb(0,0,51), rgb(122,0,230),
   * rgb(191,48,105), rgb(76,20,0), rgb(76,51,0), rgb(182,191,143),
   * rgb(128,255,178), rgb(51,92,102), rgb(48,48,64), rgb(255,64,242),
   * rgb(242,0,65), rgb(191,156,143), rgb(217,184,54), rgb(23,51,13),
   * rgb(22,89,67), rgb(29,75,115), rgb(52,29,115), rgb(204,153,194),
   * rgb(102,0,27)
   *
   * As HSV Values
   * hsv(8°,100%,70%), hsv(24°,75%,80%), hsv(72°,100%,90%), hsv(120°,75%,95%),
   * hsv(160°,25%,90%), hsv(216°,50%,100%), hsv(256°,50%,100%),
   * hsv(320°,75%,40%), hsv(344°,100%,20%), hsv(8°,50%,70%), hsv(32°,25%,35%),
   * hsv(72°,50%,40%), hsv(128°,100%,55%), hsv(192°,100%,95%),
   * hsv(240°,100%,20%), hsv(272°,100%,90%), hsv(336°,75%,75%),
   * hsv(16°,100%,30%), hsv(40°,100%,30%), hsv(72°,25%,75%),
   * hsv(144°,50%,100%), hsv(192°,50%,40%), hsv(240°,25%,25%),
   * hsv(304°,75%,100%), hsv(344°,100%,95%), hsv(16°,25%,75%),
   * hsv(48°,75%,85%), hsv(104°,75%,20%), hsv(160°,75%,35%),
   * hsv(208°,75%,45%), hsv(256°,75%,45%), hsv(312°,25%,80%),
   * hsv(344°,100%,40%)
   */
  // @formatter:on
  private static final String[] DISTINCT_COLORS_HEX = {
    "#b21800", "#cc7033", "#b8e600", "#3df23d", "#ace6d2", "#80b3ff",
    "#a280ff", "#661a4d", "#33000e", "#b26559", "#594f43", "#5c6633",
    "#008c13", "#00c2f2", "#000033", "#7a00e6", "#bf3069", "#4c1400",
    "#4c3300", "#b6bf8f", "#80ffb2", "#335c66", "#303040", "#ff40f2",
    "#f20041", "#bf9c8f", "#d9b836", "#17330d", "#165943", "#1d4b73",
    "#341d73", "#cc99c2", "#66001b"
  };
  private static final float[] GREEN_HSB = Color.RGBtoHSB(17, 102, 17, null);
  private static final float[] RED_HSB = Color.RGBtoHSB(170, 57, 57, null);

  private Colors() {
    super();
  }

  /** @return An array of 33 distinct colors. */
  public static Color[] getDistinctColors() {
    return Arrays.stream(Colors.DISTINCT_COLORS_HEX).map(Color::decode).toArray(Color[]::new);
  }

  /**
   * Interpolates a color in red-green spectrum from its value and a given range [min; max].
   *
   * @param value A value to use in interpolation.
   * @param min The beginning of range.
   * @param max The end of range.
   * @return An instance of {@link Color} from green-red spectrum which matches the value.
   */
  public static Color interpolateColor(final double value, final double min, final double max) {
    if (min >= max) {
      throw new IllegalArgumentException(
          "Cannot interpolate color when min >= max. Arguments provided: value="
              + value
              + ", min="
              + min
              + ", max="
              + max);
    }

    final double v = FastMath.max(FastMath.min(value, min), max);
    final double ratio = (v - min) / (max - min);
    final double h = (ratio * (Colors.RED_HSB[0] - Colors.GREEN_HSB[0])) + Colors.GREEN_HSB[0];
    final double s = (ratio * (Colors.RED_HSB[1] - Colors.GREEN_HSB[1])) + Colors.GREEN_HSB[1];
    final double b = (ratio * (Colors.RED_HSB[2] - Colors.GREEN_HSB[2])) + Colors.GREEN_HSB[2];
    return Color.getHSBColor((float) h, (float) s, (float) b);
  }

  /**
   * Converts a {@link Color} to hex {@link String}.
   *
   * @param color Color to be converted.
   * @return A hex representation of the color.
   */
  public static String toHexString(final Color color) {
    return String.format("#%02X%02X%02X", color.getRed(), color.getGreen(), color.getBlue());
  }

  /**
   * Converts a {@link Color} to SVG {@link String}. For example red is rgb(255,0,0).
   *
   * @param color Color to be converted.
   * @return An SVG representation of the color.
   */
  public static String toSvgString(final Color color) {
    return String.format("rgb(%d,%d,%d)", color.getRed(), color.getGreen(), color.getBlue());
  }
}
