package pl.michal.grzesiak.criticizer;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import static java.util.Optional.empty;
import static java.util.Optional.of;

public class Criticizer {

    private final ClassesProvider classesProvider;
    private final ValidationAnnotationChecker validationAnnotationChecker;
    private final ReferenceValidator referenceValidator;

    Criticizer(ClassesProvider classesProvider,
               ValidationAnnotationChecker validationAnnotationChecker, ReferenceValidator referenceValidator) {
        this.classesProvider = classesProvider;
        this.validationAnnotationChecker = validationAnnotationChecker;
        this.referenceValidator = referenceValidator;
    }

    public List<IncorrectValidation> checkAllValidationsIn(String packageName) {
        return checkAllValidationsIn(packageName, "");
    }

    public List<IncorrectValidation> checkAllValidationsIn(String packageName, String classNameSuffix) {
        List<Class<?>> classes = classesProvider.getClasses(packageName, classNameSuffix);

        return findIncorrectValidationsIn(classes);
    }

    private List<IncorrectValidation> findIncorrectValidationsIn(List<Class<?>> classes) {
        List<IncorrectValidation> incorrectValidations = new ArrayList<>();
        for (Class<?> parentClazz : classes) {
            Field[] parentClassFields = parentClazz.getDeclaredFields();
            List<String> fieldsWithLackOfValidation = findFieldsWithLackOfValidation(parentClassFields);

            if (!fieldsWithLackOfValidation.isEmpty()) {
                IncorrectValidation incorrectValidation = new IncorrectValidation(parentClazz.getCanonicalName(), fieldsWithLackOfValidation);
                incorrectValidations.add(incorrectValidation);
            }
        }
        return incorrectValidations;
    }

    private List<String> findFieldsWithLackOfValidation(Field[] parentClassFields) {
        List<String> classesWithLackOfValidation = new ArrayList<>();

        for (Field parentClassField : parentClassFields) {

            findGenericFieldWithDeadValidation(parentClassField)
                    .ifPresent(classesWithLackOfValidation::add);

            findFieldWithDeadValidation(parentClassField)
                    .ifPresent(classesWithLackOfValidation::add);
        }
        return classesWithLackOfValidation;
    }

    private Optional<String> findFieldWithDeadValidation(Field parentClassField) {
        boolean hasChildAnyValidationAnnotation = validationAnnotationChecker.hasAnyChildValidationAnnotation(parentClassField);
        if (hasChildAnyValidationAnnotation) {
            boolean isAnnotatedProperly = referenceValidator.isAnnotationPutOnReference(parentClassField);
            if (!isAnnotatedProperly) {
                return of(parentClassField.getName());
            }
        }
        return empty();
    }

    private Optional<String> findGenericFieldWithDeadValidation(Field parentClassField) {
        boolean genericField = isFieldGeneric(parentClassField.getGenericType());

        if (genericField) {
            boolean isAnnotatedProperly = referenceValidator.isAnnotationPutOnReference(parentClassField);
            if (!isAnnotatedProperly) {
                return of(parentClassField.getName());
            }
        }

        return empty();
    }

    private boolean isFieldGeneric(Type genericType) {
        if (ParameterizedType.class.isAssignableFrom(genericType.getClass())) {
            Type actualTypeArgument = ((ParameterizedType) genericType).getActualTypeArguments()[0];
            if (TypeVariable.class.isAssignableFrom(actualTypeArgument.getClass())) {
                return true;
            } else {
                isFieldGeneric(actualTypeArgument);
            }
        }
        return false;
    }
}
