/*
 * Decompiled with CFR 0.152.
 */
package pl.codewise.commons.aws;

import com.amazonaws.services.ec2.model.Instance;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import pl.codewise.commons.aws.Ec2Wrapper;
import pl.codewise.commons.aws.IpProvider;
import pl.codewise.commons.aws.Sleeper;

public class InstanceNameUpdater {
    private static final Logger log = LoggerFactory.getLogger(InstanceNameUpdater.class);
    private static final int COLLISION_RETRIES = 3;
    private final IpProvider ipProvider;
    private final String instancePrefix;
    private final Ec2Wrapper ec2Wrapper;
    private final Collection<String> autoScalingGroups;
    private final Sleeper sleeper;

    public InstanceNameUpdater(IpProvider ipProvider, String instancePrefix, Ec2Wrapper ec2Wrapper, Collection<String> autoScalingGroups, Sleeper sleeper) {
        this.ipProvider = ipProvider;
        this.instancePrefix = instancePrefix;
        this.ec2Wrapper = ec2Wrapper;
        this.autoScalingGroups = autoScalingGroups;
        this.sleeper = sleeper;
    }

    public String updateEc2InstanceName() {
        List<Instance> instances = this.getInstancesFromScalingGroups(this.autoScalingGroups);
        Instance thisInstance = this.getThisInstance(instances);
        String name = Ec2Wrapper.getInstanceName(thisInstance);
        if (StringUtils.isBlank((CharSequence)name)) {
            log.info("Setting instance name");
            this.sleeper.sleep();
            String newName = this.updateInstanceName(instances, thisInstance);
            return this.resolveNewNameIfNameCollides(newName);
        }
        log.info("Instance name is {}. No need to update.", (Object)name);
        return name;
    }

    private List<Instance> getInstancesFromScalingGroups(Collection<String> autoScalingGroups) {
        return autoScalingGroups.stream().map(this.ec2Wrapper::describeInstances).flatMap(Collection::stream).collect(Collectors.toList());
    }

    private String resolveNewNameIfNameCollides(String name) {
        String latestName = name;
        for (int i = 0; i < 3 && this.countCollidingInstances(latestName) > 1L; ++i) {
            latestName = this.updateCollidingInstanceName(latestName);
        }
        return latestName;
    }

    private long countCollidingInstances(String name) {
        return this.getInstancesFromScalingGroups(this.autoScalingGroups).stream().map(Ec2Wrapper::getInstanceName).filter(instanceName -> StringUtils.equals((CharSequence)instanceName, (CharSequence)name)).count();
    }

    private String updateCollidingInstanceName(String name) {
        log.info("Instance name collision detected for name {}, going to sleep before retry", (Object)name);
        this.sleeper.sleep();
        List<Instance> instancesAfterRandomSleep = this.getInstancesFromScalingGroups(this.autoScalingGroups);
        Instance thisInstance = this.getThisInstance(instancesAfterRandomSleep);
        return this.updateInstanceName(instancesAfterRandomSleep, thisInstance);
    }

    private String updateInstanceName(List<Instance> instances, Instance thisInstance) {
        int firstAvailableInstanceNumber = this.getFirstAvailableInstanceNumber(instances);
        String newName = this.instancePrefix + firstAvailableInstanceNumber;
        this.ec2Wrapper.setInstanceName(thisInstance, newName);
        log.info("Instance name set to {}", (Object)newName);
        return newName;
    }

    private int getFirstAvailableInstanceNumber(List<Instance> instances) {
        int highestNumber = 0;
        List instanceNumbers = instances.stream().map(Ec2Wrapper::getInstanceName).filter(instanceName -> StringUtils.startsWith((CharSequence)instanceName, (CharSequence)this.instancePrefix)).map(this::getInstanceNumber).filter(number -> number > 0).sorted().collect(Collectors.toList());
        for (Integer instanceNumber : instanceNumbers) {
            if (instanceNumber - highestNumber > 1) {
                return highestNumber + 1;
            }
            highestNumber = instanceNumber;
        }
        return highestNumber + 1;
    }

    private Integer getInstanceNumber(String instanceName) {
        String instanceNumberString = instanceName.substring(this.instancePrefix.length(), instanceName.length());
        try {
            return Integer.parseInt(instanceNumberString);
        }
        catch (NumberFormatException e) {
            log.warn("Could not get instance number from name: {}", (Object)instanceName);
            return -1;
        }
    }

    private Instance getThisInstance(List<Instance> instances) {
        String ip = this.ipProvider.getIp();
        for (Instance instance : instances) {
            if (!instance.getPublicIpAddress().equals(ip)) continue;
            return instance;
        }
        throw new RuntimeException("Could not determine Instance for IP: " + ip);
    }
}

