package pl.codewise.commons.aws.cqrs.operations.ec2;

import com.amazonaws.services.autoscaling.AmazonAutoScaling;
import com.amazonaws.services.autoscaling.model.BlockDeviceMapping;
import com.amazonaws.services.autoscaling.model.CreateLaunchConfigurationRequest;
import com.amazonaws.services.autoscaling.model.DeleteLaunchConfigurationRequest;
import com.amazonaws.services.autoscaling.model.Ebs;
import com.amazonaws.services.autoscaling.model.InstanceMonitoring;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import pl.codewise.commons.aws.cqrs.discovery.LaunchConfigurationDiscovery;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsBlockDeviceMapping;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsEbs;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsLaunchConfiguration;

import java.util.Base64;
import java.util.List;

import static java.util.stream.Collectors.toList;
import static pl.codewise.commons.aws.cqrs.utils.Functions.overrideIfNotEmpty;
import static pl.codewise.commons.aws.cqrs.utils.Functions.overrideIfNotNull;

public class LaunchConfigurationOperations {

    private static final Logger log = LoggerFactory.getLogger(LaunchConfigurationOperations.class);

    private final AmazonAutoScaling amazonAutoScaling;
    private final LaunchConfigurationDiscovery configurationDiscovery;

    public LaunchConfigurationOperations(AmazonAutoScaling amazonAutoScaling,
            LaunchConfigurationDiscovery configurationDiscovery) {
        this.amazonAutoScaling = amazonAutoScaling;
        this.configurationDiscovery = configurationDiscovery;
    }

    public AwsLaunchConfiguration copy(String sourceLaunchConfigurationName,
            AwsLaunchConfiguration newLaunchConfigurationTemplate) {
        AwsLaunchConfiguration sourceLaunchConfiguration =
                configurationDiscovery.getLaunchConfiguration(sourceLaunchConfigurationName);
        return createLaunchConfiguration(sourceLaunchConfiguration, newLaunchConfigurationTemplate);
    }

    public AwsLaunchConfiguration createLaunchConfiguration(AwsLaunchConfiguration launchConfiguration) {
        return createLaunchConfiguration(launchConfiguration, AwsLaunchConfiguration.builder().build());
    }

    private AwsLaunchConfiguration createLaunchConfiguration(AwsLaunchConfiguration sourceLaunchConfiguration,
            AwsLaunchConfiguration templateLaunchConfiguration) {
        CreateLaunchConfigurationRequest request =
                prepareCreateLaunchConfigurationRequest(sourceLaunchConfiguration, templateLaunchConfiguration);
        return createLaunchConfiguration(request);
    }

    private AwsLaunchConfiguration createLaunchConfiguration(CreateLaunchConfigurationRequest request) {
        amazonAutoScaling.createLaunchConfiguration(request);
        log.info("Launch configuration <{}> created!", request.getLaunchConfigurationName());
        return configurationDiscovery.getLaunchConfiguration(request.getLaunchConfigurationName());
    }

    public void deleteLaunchConfiguration(String launchConfigurationName) {
        DeleteLaunchConfigurationRequest request = new DeleteLaunchConfigurationRequest()
                .withLaunchConfigurationName(launchConfigurationName);
        amazonAutoScaling.deleteLaunchConfiguration(request);
        log.info("Launch configuration <{}> deleted!", request.getLaunchConfigurationName());
    }

    private CreateLaunchConfigurationRequest prepareCreateLaunchConfigurationRequest(
            AwsLaunchConfiguration sourceLaunchConfiguration, AwsLaunchConfiguration newLaunchConfigurationTemplate) {
        return new CreateLaunchConfigurationRequest()
                .withLaunchConfigurationName(
                        overrideIfNotNull(newLaunchConfigurationTemplate, sourceLaunchConfiguration,
                                AwsLaunchConfiguration::getId))
                .withImageId(overrideIfNotNull(newLaunchConfigurationTemplate, sourceLaunchConfiguration,
                        AwsLaunchConfiguration::getImageId))
                .withIamInstanceProfile(overrideIfNotNull(newLaunchConfigurationTemplate, sourceLaunchConfiguration,
                        AwsLaunchConfiguration::getIamInstanceProfile))
                .withKeyName(overrideIfNotNull(newLaunchConfigurationTemplate, sourceLaunchConfiguration,
                        AwsLaunchConfiguration::getKeyName))
                .withInstanceType(overrideIfNotNull(newLaunchConfigurationTemplate, sourceLaunchConfiguration,
                        AwsLaunchConfiguration::getInstanceType))
                .withSpotPrice(overrideIfNotNull(newLaunchConfigurationTemplate, sourceLaunchConfiguration,
                        AwsLaunchConfiguration::getSpotPrice))
                .withSecurityGroups(overrideIfNotEmpty(newLaunchConfigurationTemplate, sourceLaunchConfiguration,
                        AwsLaunchConfiguration::getSecurityGroupIds))
                .withUserData(encodeUserData(
                        overrideIfNotNull(newLaunchConfigurationTemplate, sourceLaunchConfiguration,
                                AwsLaunchConfiguration::getUserData)))
                .withAssociatePublicIpAddress(
                        overrideIfNotNull(newLaunchConfigurationTemplate, sourceLaunchConfiguration,
                                AwsLaunchConfiguration::getAssociatePublicIpAddress))
                .withInstanceMonitoring(new InstanceMonitoring().withEnabled(
                        overrideIfNotNull(newLaunchConfigurationTemplate, sourceLaunchConfiguration,
                                AwsLaunchConfiguration::getInstanceMonitoring)))
                .withBlockDeviceMappings(toBlockDeviceMappings(sourceLaunchConfiguration.getBlockDeviceMappings()));
    }

    private String encodeUserData(String userData) {
        return new String(Base64.getEncoder().encode(userData.getBytes()));
    }

    private List<BlockDeviceMapping> toBlockDeviceMappings(List<AwsBlockDeviceMapping> blockDeviceMappings) {
        return blockDeviceMappings.stream()
                .map(this::toBlockDeviceMapping)
                .collect(toList());
    }

    private BlockDeviceMapping toBlockDeviceMapping(AwsBlockDeviceMapping blockDeviceMapping) {
        return new BlockDeviceMapping()
                .withVirtualName(blockDeviceMapping.getVirtualName())
                .withDeviceName(blockDeviceMapping.getDeviceName())
                .withNoDevice(blockDeviceMapping.getNoDevice())
                .withEbs(toEbs(blockDeviceMapping.getEbs())
                );
    }

    private Ebs toEbs(AwsEbs ebs) {
        return ebs != null ?
                new Ebs()
                        .withVolumeSize(ebs.getVolumeSize())
                        .withVolumeType(ebs.getVolumeType())
                        .withDeleteOnTermination(ebs.getDeleteOnTermination())
                        .withIops(ebs.getIops())
                        .withEncrypted(ebs.getEncrypted())
                : null;
    }
}
