package pl.codewise.commons.aws.cqrs.operations.ec2;

import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.ec2.model.AllocateAddressRequest;
import com.amazonaws.services.ec2.model.AllocateAddressResult;
import com.amazonaws.services.ec2.model.AssociateAddressRequest;
import pl.codewise.commons.aws.cqrs.discovery.ec2.AddressDiscovery;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsAddress;

public class AddressOperations {

    public enum DomainType {
        EC2("standard"),
        VPC("vpc");

        private String type;

        DomainType(String type) {
            this.type = type;
        }
    }

    private final String region;
    private final AmazonEC2 amazonEC2;
    private final AddressDiscovery addressDiscovery;

    public AddressOperations(String region, AmazonEC2 amazonEC2, AddressDiscovery addressDiscovery) {
        this.region = region;
        this.amazonEC2 = amazonEC2;
        this.addressDiscovery = addressDiscovery;
    }

    public AwsAddress allocateNewAddress(DomainType domainType) {
        AllocateAddressRequest request = new AllocateAddressRequest().withDomain(domainType.type);
        AllocateAddressResult result = amazonEC2.allocateAddress(request);

        return toAwsAddress(result);
    }

    /**
     * Assign public ip to private ip on given network interface
     */
    public void associatePublicIpToPrivateIp(String publicIp, String networkInterfaceId, String privateIp) {
        AwsAddress address = getAddress(publicIp);

        AssociateAddressRequest request = new AssociateAddressRequest()
                .withAllocationId(address.getAllocationId())
                .withPrivateIpAddress(privateIp)
                .withNetworkInterfaceId(networkInterfaceId);
        amazonEC2.associateAddress(request);
    }

    private AwsAddress getAddress(String publicIp) {
        AwsAddress address = addressDiscovery.getAddressByPublicIp(publicIp);

        if (address == null) {
            throw new IllegalStateException("Failed to find Address with publicIp " + publicIp);
        }

        return address;
    }

    private AwsAddress toAwsAddress(AllocateAddressResult result) {
        return new AwsAddress.Builder()
                .withPublicIp(result.getPublicIp())
                .withRegion(region)
                .build();
    }
}
