package pl.codewise.commons.aws.cqrs.operations;

import com.amazonaws.services.route53.AmazonRoute53;
import com.amazonaws.services.route53.model.AliasTarget;
import com.amazonaws.services.route53.model.Change;
import com.amazonaws.services.route53.model.ChangeAction;
import com.amazonaws.services.route53.model.ChangeBatch;
import com.amazonaws.services.route53.model.ChangeInfo;
import com.amazonaws.services.route53.model.ChangeResourceRecordSetsRequest;
import com.amazonaws.services.route53.model.ChangeResourceRecordSetsResult;
import com.amazonaws.services.route53.model.ChangeTagsForResourceRequest;
import com.amazonaws.services.route53.model.CreateHealthCheckRequest;
import com.amazonaws.services.route53.model.CreateHealthCheckResult;
import com.amazonaws.services.route53.model.DeleteHostedZoneRequest;
import com.amazonaws.services.route53.model.DeleteHostedZoneResult;
import com.amazonaws.services.route53.model.HealthCheckConfig;
import com.amazonaws.services.route53.model.HealthCheckType;
import com.amazonaws.services.route53.model.ResourceRecord;
import com.amazonaws.services.route53.model.ResourceRecordSet;
import com.amazonaws.services.route53.model.Tag;
import com.amazonaws.services.route53.model.TagResourceType;
import com.amazonaws.services.route53.model.UpdateHealthCheckRequest;
import com.amazonaws.services.route53.model.UpdateHealthCheckResult;
import pl.codewise.commons.aws.cqrs.model.route53.AwsAliasTarget;
import pl.codewise.commons.aws.cqrs.model.route53.AwsChangeInfo;
import pl.codewise.commons.aws.cqrs.model.route53.AwsHealthCheck;
import pl.codewise.commons.aws.cqrs.model.route53.AwsRecordSet;
import pl.codewise.commons.aws.cqrs.model.route53.AwsResourceRecord;
import pl.codewise.commons.aws.cqrs.model.route53.AwsUpdateHealthCheckRequest;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.function.Function;

import static com.amazonaws.services.route53.model.ChangeAction.DELETE;
import static com.amazonaws.services.route53.model.ChangeAction.UPSERT;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;

public class Route53Operations {

    private final AmazonRoute53 amazonRoute53;

    public Route53Operations(AmazonRoute53 amazonRoute53) {
        this.amazonRoute53 = amazonRoute53;
    }

    /**
     * Create or update given DNS records
     *
     * @return change status
     */
    public AwsChangeInfo upsertChanges(String hostedZoneId, Collection<AwsRecordSet> recordSets) {
        return makeChanges(
                hostedZoneId,
                recordSets,
                (resourceRecordSet) -> toChangeOfType(UPSERT, resourceRecordSet));
    }

    /**
     * Delete given DNS records
     *
     * @return change status
     */
    public AwsChangeInfo removeRecordSets(String hostedZoneId, List<AwsRecordSet> recordSets) {
        return makeChanges(
                hostedZoneId,
                recordSets,
                (resourceRecordSet) -> toChangeOfType(DELETE, resourceRecordSet));
    }

    public AwsHealthCheck createHttpHealthCheck(String name, String ipAddress, String path,
            String callerReference, Set<String> regions, Integer failureThreshold) {

        CreateHealthCheckRequest request = new CreateHealthCheckRequest()
                .withCallerReference(callerReference)
                .withHealthCheckConfig(new HealthCheckConfig()
                        .withIPAddress(ipAddress)
                        .withPort(80)
                        .withType(HealthCheckType.HTTP)
                        .withResourcePath(path)
                        .withRegions(regions)
                        .withFailureThreshold(failureThreshold)
                );

        CreateHealthCheckResult result = amazonRoute53.createHealthCheck(request);

        ChangeTagsForResourceRequest setNameRequest = new ChangeTagsForResourceRequest()
                .withResourceType(TagResourceType.Healthcheck)
                .withResourceId(result.getHealthCheck().getId())
                .withAddTags(new Tag().withKey("Name").withValue(name));

        amazonRoute53.changeTagsForResource(setNameRequest);

        return toHealthCheck(result);
    }

    public AwsHealthCheck modifyHealthCheck(AwsUpdateHealthCheckRequest request) {

        UpdateHealthCheckRequest updateHealthCheckRequest = new UpdateHealthCheckRequest()
                .withHealthCheckId(request.getHealthCheckId())
                .withPort(request.getPort());

        UpdateHealthCheckResult result = amazonRoute53.updateHealthCheck(updateHealthCheckRequest);

        return toHealthCheck(result);
    }

    public AwsChangeInfo removeHostedZone(String hostedZoneId) {

        DeleteHostedZoneResult deleteHostedZoneResult = amazonRoute53.deleteHostedZone(
                new DeleteHostedZoneRequest().withId(hostedZoneId));

        return toAwsChangeInfo(deleteHostedZoneResult.getChangeInfo());
    }

    private AwsHealthCheck toHealthCheck(CreateHealthCheckResult result) {
        return new AwsHealthCheck(
                result.getHealthCheck().getId()
        );
    }

    private AwsHealthCheck toHealthCheck(UpdateHealthCheckResult result) {
        return new AwsHealthCheck(
                result.getHealthCheck().getId()
        );
    }

    private AwsChangeInfo makeChanges(String hostedZoneId, Collection<AwsRecordSet> recordSets,
            Function<ResourceRecordSet, Change> toChangeTypeFunction) {
        List<Change> changes = recordSets.stream()
                .map(this::toResourceRecordSet)
                .map(toChangeTypeFunction)
                .collect(toList());
        ChangeResourceRecordSetsRequest request = new ChangeResourceRecordSetsRequest()
                .withHostedZoneId(hostedZoneId)
                .withChangeBatch(new ChangeBatch().withChanges(changes));
        ChangeResourceRecordSetsResult result = amazonRoute53.changeResourceRecordSets(request);

        return toAwsChangeInfo(result.getChangeInfo());
    }

    private AwsChangeInfo toAwsChangeInfo(ChangeInfo changeInfo) {
        return new AwsChangeInfo.Builder()
                .withId(changeInfo.getId())
                .withStatus(changeInfo.getStatus())
                .build();
    }

    private Change toChangeOfType(ChangeAction changeActionType, ResourceRecordSet resourceRecordSet) {
        return new Change()
                .withAction(changeActionType)
                .withResourceRecordSet(resourceRecordSet);
    }

    private ResourceRecordSet toResourceRecordSet(AwsRecordSet recordSet) {
        return new ResourceRecordSet()
                .withName(recordSet.getName())
                .withType(recordSet.getType())
                .withTTL(recordSet.getTtl())
                .withRegion(recordSet.getRegion())
                .withSetIdentifier(recordSet.getSetIdentifier())
                .withHealthCheckId(recordSet.getHealthCheckId())
                .withMultiValueAnswer(recordSet.getMultiValueAnswer())
                .withAliasTarget(toAliasTarget(recordSet.getAliasTarget()))
                .withResourceRecords(toResourceRecords(recordSet.getResourceRecords()));
    }

    private AliasTarget toAliasTarget(AwsAliasTarget aliasTarget1) {
        return ofNullable(aliasTarget1)
                .map(aliasTarget -> {
                    AliasTarget result = new AliasTarget(aliasTarget.getHostedZoneId(), aliasTarget.getDnsName());
                    result.setEvaluateTargetHealth(aliasTarget.getEvaluateTargetHealth());
                    return result;
                }).orElse(null);
    }

    private List<ResourceRecord> toResourceRecords(List<AwsResourceRecord> resourceRecords) {
        return resourceRecords != null ?
                resourceRecords.stream()
                        .map(AwsResourceRecord::getValue)
                        .map(ResourceRecord::new)
                        .collect(toList()) :
                null;
    }
}
