package pl.codewise.commons.aws.cqrs.discovery.ec2;

import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.ec2.model.DescribeImagesRequest;
import com.amazonaws.services.ec2.model.DescribeImagesResult;
import com.amazonaws.services.ec2.model.Filter;
import com.amazonaws.services.ec2.model.Image;
import com.amazonaws.services.ec2.model.Tag;
import org.apache.commons.lang3.tuple.Pair;
import pl.codewise.commons.aws.cqrs.model.ec2.Ami;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsResourceTag;

import java.time.ZonedDateTime;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toList;

public class AmiDiscovery {

    private final AmazonEC2 amazonEC2;

    public AmiDiscovery(AmazonEC2 amazonEC2) {
        this.amazonEC2 = amazonEC2;
    }

    public Optional<Ami> findAmi(String imageId) {
        DescribeImagesRequest request = new DescribeImagesRequest()
                .withImageIds(imageId);
        return describeImages(request).findFirst();
    }

    public List<Ami> findAmisByFilters(List<Pair<String, String>> filters) {
        List<Filter> ec2Filters = filters.stream()
                .map(f -> new Filter().withName(f.getKey()).withValues(f.getValue()))
                .collect(toList());
        DescribeImagesRequest request = new DescribeImagesRequest()
                .withFilters(ec2Filters);
        return describeImages(request).collect(toList());
    }

    private Stream<Ami> describeImages(DescribeImagesRequest request) {
        DescribeImagesResult result = amazonEC2.describeImages(request);
        return result.getImages().stream()
                .map(this::toAmi);
    }

    private Ami toAmi(Image image) {
        return new Ami(
                image.getImageId(),
                image.getName(),
                ZonedDateTime.parse(image.getCreationDate()),
                image.getRootDeviceType(),
                toTags(image.getTags())
        );
    }

    private List<AwsResourceTag> toTags(List<Tag> tags) {
        return tags.stream().map(this::toTag).collect(toList());
    }

    private AwsResourceTag toTag(Tag tag) {
        return AwsResourceTag.create(tag.getKey(), tag.getValue());
    }
}
