package pl.codewise.commons.aws.cqrs.discovery.ec2;

import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.ec2.model.Address;
import com.amazonaws.services.ec2.model.DescribeAddressesRequest;
import com.amazonaws.services.ec2.model.DescribeAddressesResult;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsAddress;

import javax.annotation.Nullable;
import java.util.List;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toList;

public class AddressDiscovery {

    private final String region;
    private final AmazonEC2 amazonEC2;

    public AddressDiscovery(String region, AmazonEC2 amazonEC2) {
        this.region = region;
        this.amazonEC2 = amazonEC2;
    }

    @Nullable
    public AwsAddress getAddressByPublicIp(String publicIp) {
        DescribeAddressesRequest request = new DescribeAddressesRequest().withPublicIps(publicIp);
        DescribeAddressesResult result = amazonEC2.describeAddresses(request);

        return result.getAddresses().stream()
                .map(this::toAwsAddress)
                .findFirst()
                .orElse(null);
    }

    public List<AwsAddress> getAllAddresses() {
        return addressStream()
                .map(this::toAwsAddress)
                .collect(toList());
    }

    private AwsAddress toAwsAddress(Address address) {
        return new AwsAddress.Builder()
                .withPublicIp(address.getPublicIp())
                .withRegion(region)
                .withInstanceId(address.getInstanceId())
                .withAllocationId(address.getAllocationId())
                .withPrivateIp(address.getPrivateIpAddress())
                .withAssociationId(address.getAssociationId())
                .build();
    }

    private Stream<Address> addressStream() {
        DescribeAddressesRequest describeAddressesRequest = new DescribeAddressesRequest();
        DescribeAddressesResult result = amazonEC2.describeAddresses(describeAddressesRequest);

        return result.getAddresses().stream();
    }

}
