package pl.codewise.commons.aws.cqrs.discovery;

import com.amazonaws.services.autoscaling.AmazonAutoScaling;
import com.amazonaws.services.autoscaling.model.BlockDeviceMapping;
import com.amazonaws.services.autoscaling.model.DescribeLaunchConfigurationsRequest;
import com.amazonaws.services.autoscaling.model.DescribeLaunchConfigurationsResult;
import com.amazonaws.services.autoscaling.model.Ebs;
import com.amazonaws.services.autoscaling.model.LaunchConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsBlockDeviceMapping;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsEbs;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsLaunchConfiguration;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.function.Predicate;

import static java.util.stream.Collectors.toList;

public class LaunchConfigurationDiscovery {

    private static final Logger log = LoggerFactory.getLogger(LaunchConfigurationDiscovery.class);
    private static final Predicate<AwsLaunchConfiguration> MATCH_ALL = lc -> true;
    private static final Base64.Decoder DECODER = Base64.getDecoder();
    static final Base64.Encoder ENCODER = Base64.getEncoder();

    private final AmazonAutoScaling amazonAutoScaling;

    public LaunchConfigurationDiscovery(AmazonAutoScaling amazonAutoScaling) {
        this.amazonAutoScaling = amazonAutoScaling;
    }

    public AwsLaunchConfiguration getLaunchConfiguration(String name) {
        DescribeLaunchConfigurationsRequest request = new DescribeLaunchConfigurationsRequest()
                .withLaunchConfigurationNames(name);
        DescribeLaunchConfigurationsResult result = amazonAutoScaling.describeLaunchConfigurations(request);
        if (result.getLaunchConfigurations().isEmpty()) {
            log.warn("Could not find launch configuration named " + name);
            return null;
        }
        return toLaunchConfiguration(result.getLaunchConfigurations().get(0));
    }

    private AwsLaunchConfiguration toLaunchConfiguration(LaunchConfiguration launchConfiguration) {
        return AwsLaunchConfiguration.builder()
                .withName(launchConfiguration.getLaunchConfigurationName())
                .withImageId(launchConfiguration.getImageId())
                .withIamInstanceProfile(launchConfiguration.getIamInstanceProfile())
                .withKeyName(launchConfiguration.getKeyName())
                .withInstanceType(launchConfiguration.getInstanceType())
                .withSpotPrice(launchConfiguration.getSpotPrice())
                .withSecurityGroupIds(launchConfiguration.getSecurityGroups())
                .withUserData(decode(launchConfiguration.getUserData()))
                .withAssociatePublicIpAddress(launchConfiguration.getAssociatePublicIpAddress())
                .withInstanceMonitoring(launchConfiguration.getInstanceMonitoring() != null ?
                        launchConfiguration.getInstanceMonitoring().isEnabled() : false)
                .withBlockDeviceMappings(toBlockDeviceMappings(launchConfiguration.getBlockDeviceMappings()))
                .withCreationTime(LocalDateTime.ofInstant(launchConfiguration.getCreatedTime().toInstant(), ZoneId.systemDefault()))
                .build();
    }

    private List<AwsBlockDeviceMapping> toBlockDeviceMappings(List<BlockDeviceMapping> blockDeviceMappings) {
        return blockDeviceMappings.stream().map(this::toBlockDeviceMapping).collect(toList());
    }

    private AwsBlockDeviceMapping toBlockDeviceMapping(BlockDeviceMapping blockDeviceMapping) {
        return new AwsBlockDeviceMapping(
                blockDeviceMapping.getVirtualName(),
                blockDeviceMapping.getDeviceName(),
                blockDeviceMapping.getNoDevice(),
                toEbs(blockDeviceMapping.getEbs())
        );
    }

    private AwsEbs toEbs(Ebs ebs) {
        return ebs != null ? new AwsEbs(
                ebs.getSnapshotId(),
                ebs.getVolumeSize(),
                ebs.getVolumeType(),
                ebs.getDeleteOnTermination(),
                ebs.getIops(),
                ebs.getEncrypted()
        ) : null;
    }

    public List<AwsLaunchConfiguration> listAllLaunchConfigurations() {
        return listLaunchConfigurationsByPredicate(MATCH_ALL);
    }

    public List<AwsLaunchConfiguration> listLaunchConfigurationsByPredicate(
            Predicate<AwsLaunchConfiguration> predicate) {
        DescribeLaunchConfigurationsResult result;
        String nextToken = null;
        List<AwsLaunchConfiguration> launchConfigurations = new ArrayList<>();
        do {
            result = amazonAutoScaling.describeLaunchConfigurations(
                    new DescribeLaunchConfigurationsRequest()
                            .withNextToken(nextToken)
            );

            List<AwsLaunchConfiguration> matching = result.getLaunchConfigurations().stream()
                    .map(this::toLaunchConfiguration)
                    .filter(predicate)
                    .collect(toList());
            launchConfigurations.addAll(matching);
        } while ((nextToken = result.getNextToken()) != null);

        return launchConfigurations;
    }

    private String decode(String userData) {
        return userData != null ? new String(DECODER.decode(userData)) : null;
    }
}
