package pl.codewise.commons.aws.cqrs.discovery;

import com.amazonaws.services.identitymanagement.AmazonIdentityManagement;
import com.amazonaws.services.identitymanagement.model.AmazonIdentityManagementException;
import com.amazonaws.services.identitymanagement.model.AttachedPolicy;
import com.amazonaws.services.identitymanagement.model.GetInstanceProfileRequest;
import com.amazonaws.services.identitymanagement.model.GetInstanceProfileResult;
import com.amazonaws.services.identitymanagement.model.GetRoleRequest;
import com.amazonaws.services.identitymanagement.model.GetRoleResult;
import com.amazonaws.services.identitymanagement.model.ListAttachedRolePoliciesRequest;
import com.amazonaws.services.identitymanagement.model.ListAttachedRolePoliciesResult;
import com.amazonaws.services.identitymanagement.model.ListPoliciesRequest;
import com.amazonaws.services.identitymanagement.model.ListPoliciesResult;
import com.amazonaws.services.identitymanagement.model.ListServerCertificatesResult;
import com.amazonaws.services.identitymanagement.model.Policy;
import com.amazonaws.services.identitymanagement.model.Role;
import com.amazonaws.services.identitymanagement.model.ServerCertificateMetadata;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

public class IamDiscovery {

    private final AmazonIdentityManagement amazonIdentityManagement;

    public IamDiscovery(AmazonIdentityManagement amazonIdentityManagement) {
        this.amazonIdentityManagement = amazonIdentityManagement;
    }

    public List<String> listAccountAliases() {
        return amazonIdentityManagement.listAccountAliases().getAccountAliases();
    }

    public boolean hasRole(String roleName) {
        return getRoleArn(roleName).isPresent();
    }

    public Optional<String> getRoleArn(String roleName) {
        try {
            GetRoleRequest request = new GetRoleRequest()
                    .withRoleName(roleName);
            GetRoleResult result = amazonIdentityManagement.getRole(request);
            return Optional.of(result.getRole().getArn());
        } catch (AmazonIdentityManagementException e) {
            return Optional.empty();
        }
    }

    public boolean hasInstanceProfile(String instanceProfileName) {
        return getInstanceProfileArn(instanceProfileName).isPresent();
    }

    public Optional<String> getInstanceProfileArn(String instanceProfileName) {
        try {
            GetInstanceProfileRequest request = new GetInstanceProfileRequest()
                    .withInstanceProfileName(instanceProfileName);
            GetInstanceProfileResult result = amazonIdentityManagement.getInstanceProfile(request);
            return Optional.of(result.getInstanceProfile().getArn());
        } catch (AmazonIdentityManagementException e) {
            return Optional.empty();
        }
    }

    public boolean hasPolicy(String name) {
        return getPolicyArn(name).isPresent();
    }

    public Optional<String> getPolicyArn(String policyName) {
        String marker = null;
        Optional<String> policy;
        ListPoliciesResult result;
        do {
            result = amazonIdentityManagement.listPolicies(new ListPoliciesRequest()
                    .withMarker(marker));
            policy = result.getPolicies().stream()
                    .filter(p -> policyName.equals(p.getPolicyName()))
                    .findFirst()
                    .map(Policy::getArn);
            marker = result.getMarker();
        }
        while (!policy.isPresent() && result.isTruncated());
        return policy;
    }

    public List<String> getAttachedPoliciesArnsForRole(String roleName) {
        ListAttachedRolePoliciesRequest request = new ListAttachedRolePoliciesRequest().withRoleName(roleName);
        try {
            ListAttachedRolePoliciesResult result = amazonIdentityManagement.listAttachedRolePolicies(request);
            return result.getAttachedPolicies().stream()
                    .map(AttachedPolicy::getPolicyArn)
                    .collect(Collectors.toList());
        } catch (AmazonIdentityManagementException e) {
            return Collections.emptyList();
        }
    }

    public List<String> getInstanceProfileRoles(String profileName) {
        GetInstanceProfileRequest request = new GetInstanceProfileRequest().withInstanceProfileName(profileName);
        try {
            GetInstanceProfileResult result = amazonIdentityManagement.getInstanceProfile(request);
            return result.getInstanceProfile().getRoles().stream()
                    .map(Role::getRoleName)
                    .collect(Collectors.toList());
        } catch (AmazonIdentityManagementException e) {
            return Collections.emptyList();
        }
    }

    public Optional<String> getServerCertificateArn(String certificateName) {
        ListServerCertificatesResult result = amazonIdentityManagement.listServerCertificates();
        return result.getServerCertificateMetadataList().stream()
                .filter(serviceCertificate ->
                        Objects.equals(certificateName, serviceCertificate.getServerCertificateName()))
                .map(ServerCertificateMetadata::getArn)
                .findFirst();
    }
}
