package pl.codewise.commons.aws.cqrs.discovery;

import com.amazonaws.services.elasticloadbalancing.AmazonElasticLoadBalancing;
import com.amazonaws.services.elasticloadbalancing.model.DescribeInstanceHealthRequest;
import com.amazonaws.services.elasticloadbalancing.model.DescribeInstanceHealthResult;
import com.amazonaws.services.elasticloadbalancing.model.DescribeLoadBalancersRequest;
import com.amazonaws.services.elasticloadbalancing.model.DescribeLoadBalancersResult;
import com.amazonaws.services.elasticloadbalancing.model.Instance;
import com.amazonaws.services.elasticloadbalancing.model.InstanceState;
import com.amazonaws.services.elasticloadbalancing.model.LoadBalancerDescription;
import pl.codewise.commons.aws.cqrs.model.AwsInstance;

import java.util.List;

import static java.util.stream.Collectors.toList;

public class ClassicLoadBalancingDiscovery {

    public static final String IN_SERVICE_INSTANCE_STATE = "InService";
    public static final String OUT_OF_SERVICE_INSTANCE_STATE = "OutOfService";
    public static final String UNKNOWN_INSTANCE_STATE = "Unknown";

    private final AmazonElasticLoadBalancing amazonElasticLoadBalancing;

    public ClassicLoadBalancingDiscovery(AmazonElasticLoadBalancing amazonElasticLoadBalancing) {
        this.amazonElasticLoadBalancing = amazonElasticLoadBalancing;
    }

    public boolean hasLoadBalancer(String loadBalancerName) {
        DescribeLoadBalancersRequest request =
                new DescribeLoadBalancersRequest().withLoadBalancerNames(loadBalancerName);
        DescribeLoadBalancersResult result =
                amazonElasticLoadBalancing.describeLoadBalancers(request);
        return !result.getLoadBalancerDescriptions().isEmpty();
    }

    public boolean areInstancesInService(String loadBalancerName, List<AwsInstance> awsInstances) {
        return areInstancesInState(loadBalancerName, IN_SERVICE_INSTANCE_STATE, awsInstancesToInstances(awsInstances));
    }

    public boolean areInstanceIdsInService(String loadBalancerName, List<String> instanceIds) {
        return areInstancesInState(loadBalancerName, IN_SERVICE_INSTANCE_STATE, instanceIdsToInstances(instanceIds));
    }

    public boolean areInstancesOutOfService(String loadBalancerName, List<AwsInstance> awsInstances) {
        return areInstancesInState(loadBalancerName, OUT_OF_SERVICE_INSTANCE_STATE,
                awsInstancesToInstances(awsInstances));
    }

    private boolean areInstancesInState(String loadBalancerName, String state, List<Instance> instances) {
        DescribeInstanceHealthRequest request = new DescribeInstanceHealthRequest(loadBalancerName)
                .withInstances(instances);
        DescribeInstanceHealthResult result = amazonElasticLoadBalancing.describeInstanceHealth(request);
        List<InstanceState> instanceStates = result.getInstanceStates();
        return instanceStates.size() == instances.size() &&
                instanceStates.stream().allMatch(is -> is.getState().equals(state));
    }

    public LoadBalancerDescription getLoadBalancer(String loadBalancerName) {
        DescribeLoadBalancersRequest request =
                new DescribeLoadBalancersRequest().withLoadBalancerNames(loadBalancerName);
        DescribeLoadBalancersResult result = amazonElasticLoadBalancing.describeLoadBalancers(request);
        if (result.getLoadBalancerDescriptions().isEmpty()) {
            return null;
        }
        return result.getLoadBalancerDescriptions().get(0);
    }

    private List<Instance> awsInstancesToInstances(List<AwsInstance> awsInstances) {
        return awsInstances.stream()
                .map(AwsInstance::getInstanceId)
                .map(Instance::new)
                .collect(toList());
    }

    private List<Instance> instanceIdsToInstances(List<String> instanceIds) {
        return instanceIds.stream()
                .map(Instance::new)
                .collect(toList());
    }
}
