package pl.codewise.commons.aws;

import com.amazonaws.services.autoscaling.AmazonAutoScaling;
import com.amazonaws.services.cloudwatch.AmazonCloudWatch;
import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.elasticloadbalancing.AmazonElasticLoadBalancing;
import com.amazonaws.services.identitymanagement.AmazonIdentityManagement;
import com.amazonaws.services.lambda.AWSLambda;
import com.amazonaws.services.route53.AmazonRoute53;
import com.amazonaws.services.route53domains.AmazonRoute53Domains;
import com.amazonaws.services.s3.AmazonS3;
import pl.codewise.commons.aws.cqrs.factories.AmazonClientBuilders;
import pl.codewise.commons.aws.cqrs.factories.AssumeRoleParameters;
import pl.codewise.commons.aws.cqrs.factories.internal.Builder;

import java.util.concurrent.ConcurrentHashMap;

public class AwsClientsCache {

    private final ConcurrentHashMap<String, AmazonEC2> amazonEc2Cache = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, AmazonAutoScaling> amazonAutoScalingCache = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, AWSLambda> awsLambdaCache = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, AmazonElasticLoadBalancing> amazonElasticLoadBalancingCache =
            new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, com.amazonaws.services.elasticloadbalancingv2.AmazonElasticLoadBalancing>
            applicationLoadBalancingCache = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, AmazonCloudWatch> amazonCloudWatchCache = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, AmazonRoute53> amazonRoute53Cache = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, AmazonRoute53Domains> amazonRoute53DomainsCache = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, AmazonIdentityManagement> amazonIdentityManagementCache =
            new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, AmazonS3> amazonS3Cache = new ConcurrentHashMap<>();

    private final AmazonClientBuilders amazonClientBuilders;

    public AwsClientsCache(AmazonClientBuilders amazonClientBuilders) {
        this.amazonClientBuilders = amazonClientBuilders;
    }

    public static <T> T awsClient(AssumeRoleParameters parameters, Builder<T> builder) {
        return builder.withAssumeRoleParameters(parameters).get();
    }

    public static <T> T awsClient(String region, Builder<T> builder) {
        return builder.withRegion(region).get();
    }

    public static <T> T awsClient(String region, AssumeRoleParameters parameters, Builder<T> builder) {
        return builder.withRegion(region).withAssumeRoleParameters(parameters).get();
    }

    public AmazonS3 newAmazonS3() {
        return amazonS3Cache.computeIfAbsent("DEFAULT_S3_CLIENT", k -> awsClient(amazonClientBuilders.s3()));
    }

    public AmazonS3 newAmazonS3(AssumeRoleParameters parameters) {
        return amazonS3Cache.computeIfAbsent(cacheKey("S3_CLIENT", parameters),
                k -> awsClient(parameters, amazonClientBuilders.s3()));
    }

    public AmazonIdentityManagement newAmazonIdentityManagement() {
        return amazonIdentityManagementCache
                .computeIfAbsent("DEFAULT_IAM_CLIENT", k -> awsClient(amazonClientBuilders.iam()));
    }

    public AmazonIdentityManagement newAmazonIdentityManagement(AssumeRoleParameters parameters) {
        return amazonIdentityManagementCache.computeIfAbsent(cacheKey("IAM_CLIENT", parameters),
                k -> awsClient(parameters, amazonClientBuilders.iam()));
    }

    public AmazonRoute53 newAmazonRoute53() {
        return amazonRoute53Cache
                .computeIfAbsent("DEFAULT_ROUTE53_CLIENT", k -> awsClient(amazonClientBuilders.route53()));
    }

    public AmazonRoute53 newAmazonRoute53(AssumeRoleParameters parameters) {
        String cacheKey = cacheKey("ROUTE53_CLIENT", parameters);
        return amazonRoute53Cache.computeIfAbsent(cacheKey, k -> awsClient(parameters, amazonClientBuilders.route53()));
    }

    public AmazonRoute53Domains newAmazonRoute53Domains(AssumeRoleParameters parameters) {
        String cacheKey = cacheKey("ROUTE53DOMAINS_CLIENT", parameters);
        return amazonRoute53DomainsCache
                .computeIfAbsent(cacheKey, k -> awsClient(parameters, amazonClientBuilders.route53Domains()));
    }

    public AmazonRoute53Domains newAmazonRoute53Domains() {
        return amazonRoute53DomainsCache.computeIfAbsent("DEFAULT_ROUTE53DOMAINS_CLIENT",
                k -> awsClient(amazonClientBuilders.route53Domains()));
    }

    public AmazonEC2 newAmazonEC2(String region) {
        return amazonEc2Cache.computeIfAbsent(region, k -> awsClient(region, amazonClientBuilders.ec2()));
    }

    public AmazonEC2 newAmazonEC2(String region, AssumeRoleParameters parameters) {
        return amazonEc2Cache.computeIfAbsent(cacheKey(region, parameters),
                k -> awsClient(region, parameters, amazonClientBuilders.ec2()));
    }

    public AmazonAutoScaling newAmazonAutoScaling(String region) {
        return amazonAutoScalingCache
                .computeIfAbsent(region, k -> awsClient(region, amazonClientBuilders.autoScaling()));
    }

    public AmazonAutoScaling newAmazonAutoScaling(String region, AssumeRoleParameters parameters) {
        return amazonAutoScalingCache.computeIfAbsent(cacheKey(region, parameters),
                k -> awsClient(region, parameters, amazonClientBuilders.autoScaling()));
    }

    public AWSLambda newAwsLambda(String region) {
        return awsLambdaCache.computeIfAbsent(region, k -> awsClient(region, amazonClientBuilders.awsLambda()));
    }

    public AWSLambda newAwsLambda(String region, AssumeRoleParameters parameters) {
        return awsLambdaCache.computeIfAbsent(cacheKey(region, parameters),
                k -> awsClient(region, parameters, amazonClientBuilders.awsLambda()));
    }

    public AmazonElasticLoadBalancing newAmazonElasticLoadBalancing(String region) {
        return amazonElasticLoadBalancingCache
                .computeIfAbsent(region, k -> awsClient(region, amazonClientBuilders.amazonElasticLoadBalancing()));
    }

    public AmazonElasticLoadBalancing newAmazonElasticLoadBalancing(String region, AssumeRoleParameters parameters) {
        return amazonElasticLoadBalancingCache.computeIfAbsent(cacheKey(region, parameters),
                k -> awsClient(region, parameters, amazonClientBuilders.amazonElasticLoadBalancing()));
    }

    public com.amazonaws.services.elasticloadbalancingv2.AmazonElasticLoadBalancing newApplicationLoadBalancing(
            String region) {
        return applicationLoadBalancingCache
                .computeIfAbsent(region, k -> awsClient(region, amazonClientBuilders.applicationLoadBalancing()));
    }

    public com.amazonaws.services.elasticloadbalancingv2.AmazonElasticLoadBalancing newApplicationLoadBalancing(
            String region, AssumeRoleParameters parameters) {
        return applicationLoadBalancingCache.computeIfAbsent(cacheKey(region, parameters),
                k -> awsClient(region, parameters, amazonClientBuilders.applicationLoadBalancing()));
    }

    public AmazonCloudWatch newAmazonCloudWatch(String region) {
        return amazonCloudWatchCache
                .computeIfAbsent(region, k -> awsClient(region, amazonClientBuilders.amazonCloudWatch()));
    }

    public AmazonCloudWatch newAmazonCloudWatch(String region, AssumeRoleParameters parameters) {
        return amazonCloudWatchCache.computeIfAbsent(cacheKey(region, parameters),
                k -> awsClient(region, parameters, amazonClientBuilders.amazonCloudWatch()));
    }

    private <T> T awsClient(Builder<T> builder) {
        return builder.get();
    }

    private String cacheKey(String prefix, AssumeRoleParameters parameters) {
        return prefix + "-" + parameters.getRoleArn();
    }
}
