package net.yakclient.web.utils.helper

inline fun <reified E : Enum<E>> valueOf(name: String, ignoreCase : Boolean = true): E? =
    valueOf<E, String>(name) { lookFor, enum -> enum.name.equals(lookFor, ignoreCase) }

inline fun <reified E : Enum<E>> valueOf(name: String, crossinline error: () -> Any): E = checkNotNull(valueOf<E>(name), error)

/**
 * Attempts to find the value of the generified enum given a key, K, and a function to
 * map Enum types to the given key. When the first successful match of the mapper is found
 * this function will return. If no match is found null is returned.
 *
 * Example usage:
 *
 * ```
 * enum class YourEnum(
 *     val name: String
 * ) {
 *     VALUE_ONE("Bob"),
 *     ...;
 * }
 *
 * ...
 *
 * val myEnum = valueOf<YourEnum>("bOB") { bob : String, enumValue : YourEnum->
 *     enumValue.name.equals(bob, ignoreCase = true)
 * }
 *
 * assertNotNull(myEnum) { "Bob not found?!" }
 * ```
 *
 * @param key The key to look for when mapping
 * @param mapper The function to map by.
 * @return The matched enum or null if it cannot be found by the given key.
 *
 */
inline fun <reified E : Enum<E>, K> valueOf(key: K, mapper: (K, E) -> Boolean): E? {
    val values = E::class.java.enumConstants
    for (value in values) {
        if (mapper(key, value)) return value
    }
    return null
}

fun <E : Enum<E>> E.or(vararg type: Enum<E>): Boolean = type.any { equals(it) }

fun <E : Enum<E>> E.and(vararg type: Enum<E>): Boolean = type.all { equals(it) }
