package net.yakclient.web.utils.helper

import org.springframework.web.util.WebUtils
import java.net.HttpCookie
import javax.servlet.http.Cookie
import javax.servlet.http.HttpServletRequest

/**
 * A convenience method to find the first cookie matching the given name.
 * This utils function delegates the actual implementation to the Webutils
 * convenience class.
 *
 *
 * @receiver The request to find the cookie in.
 * @param name The name of the cookie to find.
 * @return The cookie with the specified name, or null if it cannot be found.
 *
 * @see WebUtils
 */
fun HttpServletRequest.findCookie(name: String): Cookie? = WebUtils.getCookie(this, name)

/**
 * Parses all cookies from a given header. Under the hood this method uses the
 * java.net.HttpCookie to parse the header, and from there maps each to a
 * javax.servlet.Cookie.
 *
 * @param header The header to parse cookies from.
 * @return The cookies that have been parsed.
 *
 * @throws IllegalArgumentException If an error occurs during parsing.
 *
 * @see HttpCookie
 * @see Cookie
 */
@Throws(IllegalArgumentException::class)
fun parseCookies(header: String): List<Cookie> =
    HttpCookie.parse(header).map(HttpCookie::toServletCookie)

fun parseCookies(headers: List<String>): List<Cookie> =
    headers.flatMap { HttpCookie.parse(it).map(HttpCookie::toServletCookie) }

private fun HttpCookie.toServletCookie(): Cookie = Cookie(name, value).apply {
    val cookie = this@toServletCookie

    fun <T> T?.set(block: (T) -> Unit) = if (this != null) block(this) else Unit

    cookie.comment.set { comment = it }
    cookie.domain.set { domain = it }
    cookie.maxAge.set { maxAge = it.toInt() }
    cookie.path.set { path = it }
    cookie.version.set { version = it }
    cookie.isHttpOnly.set { isHttpOnly = it }
}