package net.yakclient.web.utils.helper

import net.yakclient.web.utils.config.ConfigReader
import net.yakclient.web.utils.config.DomainDelegatingConfig
import net.yakclient.web.utils.config.FileConfigReader
import org.springframework.core.io.Resource
import java.io.File

/**
 * Caches Config readers and prevents reloading of files.
 *
 * @property configs Cached configs stored by the absolute file name.
 *
 * @see ConfigReader
 */
object ConfigHelper {
    private val configs: MutableMap<String, ConfigReader> = HashMap()

    /**
     * Retrieves or creates a config reader for the given file.
     *
     * @param file The file to find a ConfigReader for.
     * @return The config reader for the given file
     *
     * @see FileConfigReader
     */
    fun getConfig(file: File): ConfigReader = configs.getOrPut(file.absolutePath) { FileConfigReader(file) }

    /**
     * Returns a config based on the location of the given string.
     *
     * @param file The string to find a config reader by.
     * @return The config reader for the given string.
     */
    fun getConfig(file: String) : ConfigReader = getConfig(File(file))

    /**
     * Returns a config reader based on the spring resource given.
     *
     * @param resource The resource to find a config on.
     * @return The config reader for the given resource.
     *
     * @see Resource
     */
    fun getConfig(resource: Resource) : ConfigReader = getConfig(resource.file)
}

/**
 * Returns a config reader that has a set domain and all queries will
 * be filtered through this domain before being delegated to the parent.
 *
 * @sample useDomain with a domain of "net.yakclient" this, "net.yakclient.something.cool" becomes "something.cool"
 *
 * Example usage
 * ```
 * val config: ConfigReader = ...
 * val delegatedConfig = config.useDomain("could.be.something.longer")
 *
 * assertEquals(config.readString("could.be.something.longer.test-string"), delegatedConfig.readString("test-string"))
 *
 * println("It worked!)
 * ```
 *
 * Defining a custom separator:
 *
 * ```
 * val config: ConfigReader = ...
 * val delegatedConfig = config.useDomain("could.be.something.longer", '#' /* Any char you want */)
 * ```
 *
 * @receiver The config reader to apply a domain to.
 * @param domain Required, the domain to apply.
 * @param separator Not required, the separator to use. This defaults to '.'
 *
 * @return The config reader delegating all calls through the domain and to the receiver.
 */
fun ConfigReader.useDomain(
    domain: String,
    separator: Char = '.'
) : ConfigReader = DomainDelegatingConfig(domain, separator, this)


