package net.wisedream.ezhc;

import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;

import java.io.IOException;

/**
 * Stateless HttpClientService executes {@link org.apache.http.client.methods.HttpUriRequest}s
 * separately by using distinct {@link org.apache.http.client.protocol.HttpClientContext}.
 *
 * @see net.wisedream.ezhc.StatefulHttpClientService
 */
public class HttpClientService {
    private CloseableHttpClient httpClient;
    private RequestCustomizer customizer;

    public HttpClientService(CloseableHttpClient httpClient) {
        this.httpClient = httpClient;
    }

    public HttpClientService(CloseableHttpClient httpClient, RequestCustomizer customizer) {
        this(httpClient);
        this.customizer = customizer;
    }

    /**
     * Release underlying resources.
     *
     * @throws java.io.IOException
     */
    public void dispose() throws IOException {
        this.httpClient.close();
    }


    /**
     * Send request and reads the response content
     *
     * @param req
     * @return null if failed
     */
    public String request(HttpUriRequest req, HttpClientContext cxt, String defaultCharset) {
        String content = null;
        try (CloseableHttpResponse resp = this.execute(req, cxt)) {
            content = EntityUtils.toString(resp.getEntity(), defaultCharset == null ? "UTF-8" : defaultCharset);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return content;
    }

    public CloseableHttpResponse execute(HttpUriRequest request, HttpClientContext cxt) throws IOException {
        return httpClient.execute(this.customizer == null ? request : customizer.custom(request), cxt);
    }

    /**
     * Make some modifications to all HttpUriRequests before execution.
     */
    public interface RequestCustomizer {
        public HttpUriRequest custom(final HttpUriRequest request);
    }
}
