package net.wisedream.ezhc;

import net.wisedream.ezhc.bean.HttpClient;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CookieStore;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpRequestWrapper;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.*;
import org.apache.http.impl.conn.DefaultProxyRoutePlanner;
import org.apache.http.impl.cookie.BasicClientCookie;
import org.apache.http.message.BasicHeader;
import org.apache.http.protocol.HttpContext;
import org.apache.http.ssl.SSLContextBuilder;

import java.io.IOException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Created by pseudo on 16-1-26.
 */
public abstract class HttpClientFactory {
    protected final CloseableHttpClient create(HttpClient hcBean) throws KeyStoreException, NoSuchAlgorithmException, KeyManagementException {
        HttpClientBuilder builder = HttpClients.custom();
        //lower-level proxy
        if (hcBean.getProxy() != null) {
            HttpHost proxy = new HttpHost(hcBean.getProxy().getHost(), hcBean.getProxy().getPort().intValue());
            builder.setRoutePlanner(new DefaultProxyRoutePlanner(proxy));
        }
        //timeout
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(hcBean.getConnectTimeout().intValue())
                .setSocketTimeout(hcBean.getSocketTimeout().intValue())
                .setConnectionRequestTimeout(hcBean.getConnRequestTimeout().intValue())
                .setMaxRedirects(10)
                .build();
        builder.setDefaultRequestConfig(requestConfig);
        //credential
        if (hcBean.getCredentials() != null) {
            CredentialsProvider credsProvider = new BasicCredentialsProvider();
            List<HttpClient.Credentials.Credential> credentials = hcBean.getCredentials().getCredential();
            for (HttpClient.Credentials.Credential credential : credentials) {
                final UsernamePasswordCredentials usernamePassword = new UsernamePasswordCredentials(credential.getUsername(), credential.getPassword());
                credential.getScope().forEach(ele -> credsProvider.setCredentials(new AuthScope(ele.getHost(), ele.getPort().intValue()), usernamePassword));
            }
            builder.setProxyAuthenticationStrategy(new ProxyAuthenticationStrategy()).setDefaultCredentialsProvider(credsProvider);
        }
        //user-agent
        if (hcBean.getUserAgent() != null) {
            builder.setUserAgent(hcBean.getUserAgent().trim());
        }
        //cookie store
        if (hcBean.getCookieStore() != null) {
            CookieStore store = new BasicCookieStore();
            hcBean.getCookieStore().getCookie().forEach(ele -> {
                BasicClientCookie cookie = new BasicClientCookie(ele.getName(), ele.getValue());
                cookie.setDomain(ele.getDomain());
                cookie.setPath(ele.getPath());
                store.addCookie(cookie);
            });
            builder.setDefaultCookieStore(store);
        }
        //default header
        if (hcBean.getDefaultHeaders() != null) {
            List<BasicHeader> headers = hcBean.getDefaultHeaders().getHeader().stream().map(ele ->
                    new BasicHeader(ele.getName(), ele.getValue())).collect(Collectors.toList());
            builder.setDefaultHeaders(headers);
        }
        //specific header
        if (hcBean.getSpecificHeaders() != null)
            builder.addInterceptorFirst(new HttpRequestInterceptor() {
                private List<HttpClient.SpecificHeaders.Header> headers;

                {
                    this.headers = hcBean.getSpecificHeaders().getHeader();
                }

                @Override
                public void process(HttpRequest request, HttpContext context) throws HttpException, IOException {
                    String url = extractUrl(request);
                    headers.stream()
                            .filter(header -> url.matches(header.getUrlMatches()) && request.getFirstHeader(header.getName()) == null)
                            .forEach(header -> request.addHeader(header.getName(), header.getValue()));
                }
            });
        return builder.setSSLContext(new SSLContextBuilder().loadTrustMaterial(null, (chain, authType) -> true).build()).build();
    }

    private String extractUrl(HttpRequest request) {
        if (request instanceof HttpRequestWrapper)
            return extractUrl(((HttpRequestWrapper) request).getOriginal());
        else
            return ((HttpUriRequest) request).getURI().toString();
    }

    public abstract CloseableHttpClient create() throws Exception;
}
