/*
 * Copyright (c) 2007-2022 The Cascading Authors. All Rights Reserved.
 *
 * Project and contact information: https://cascading.wensel.net/
 *
 * This file is part of the Cascading project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cascading.operation.expression;

import java.beans.ConstructorProperties;
import java.io.IOException;
import java.io.StringReader;

import cascading.operation.OperationException;
import cascading.tuple.Fields;
import org.codehaus.commons.compiler.CompileException;
import org.codehaus.janino.ExpressionEvaluator;
import org.codehaus.janino.Scanner;

import static cascading.tuple.coerce.Coercions.asClass;

/**
 * Class ExpressionOperation is the base class for {@link ExpressionFunction}, {@link ExpressionFilter},
 * {@link cascading.operation.assertion.AssertExpression}.
 */
public class ExpressionOperation extends ScriptOperation
  {
  @ConstructorProperties({"expression"})
  protected ExpressionOperation( String expression )
    {
    super( ANY, expression, Boolean.class );
    }

  @ConstructorProperties({"fieldDeclaration", "expression"})
  protected ExpressionOperation( Fields fieldDeclaration, String expression )
    {
    super( ANY, fieldDeclaration, expression, asClass( fieldDeclaration.getType( 0 ) ) );
    }

  @ConstructorProperties({"fieldDeclaration", "expression", "parameterType"})
  protected ExpressionOperation( Fields fieldDeclaration, String expression, Class parameterType )
    {
    super( 1, fieldDeclaration, expression, asClass( fieldDeclaration.getType( 0 ) ), null,
      new Class[]{parameterType} );
    }

  @ConstructorProperties({"fieldDeclaration", "expression", "parameterNames", "parameterTypes"})
  protected ExpressionOperation( Fields fieldDeclaration, String expression, String[] parameterNames, Class[] parameterTypes )
    {
    super( parameterTypes.length, fieldDeclaration, expression, asClass( fieldDeclaration.getType( 0 ) ), parameterNames, parameterTypes );
    }

  @ConstructorProperties({"expression", "parameterType"})
  protected ExpressionOperation( String expression, Class parameterType )
    {
    super( 1, expression, Object.class, null, new Class[]{parameterType} );
    }

  @ConstructorProperties({"expression", "parameterNames", "parameterTypes"})
  protected ExpressionOperation( String expression, String[] parameterNames, Class[] parameterTypes )
    {
    super( parameterTypes.length, expression, Object.class, parameterNames, parameterTypes );
    }

  public String getExpression()
    {
    return getBlock();
    }

  protected String[] guessParameterNames() throws CompileException, IOException
    {
    return ExpressionEvaluator.guessParameterNames( new Scanner( "expressionEval", new StringReader( block ) ) );
    }

  @Override
  protected Evaluator getEvaluator( Class returnType, String[] parameterNames, Class[] parameterTypes )
    {
    try
      {
      ExpressionEvaluator evaluator = new ExpressionEvaluator();
      evaluator.setReturnType( returnType );
      evaluator.setParameters( parameterNames, parameterTypes );
      evaluator.setExtendedClass( getExtendedClass() );
      evaluator.cook( block );

      return evaluator::evaluate;
      }
    catch( CompileException exception )
      {
      throw new OperationException( "could not compile expression: " + block, exception );
      }
    }
  }
