/*
 * Copyright (c) 2012, Johannes Rudolph
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package net.virtualvoid.codec

/**
 * An encoder is a "function" from `I` to `O`. It may be not a function in the mathematical
 * sense in that it may return different values for the same input. For example, an encryption
 * function may use a different random factor for each call of encode.
 *
 * @tparam I
 * @tparam O
 */
trait Encoder[-I, +O] { self =>
  /**
   * The encoding function. Returns either `Right(code)` if encoding is possible
   * for the input value with `code` being the encoded value or `Left(exception)` if
   * the input value can't be encoded.
   *
   * @param i
   * @return
   */
  def encode(i: I): OrError[O]

  def ~>[O2](next: Encoder[O, O2]): Encoder[I, O2] =
    new Encoder[I, O2] {
      val func = chain(self, next)
      def encode(i: I): OrError[O2] =
        func(i)
    }
}

object Encoder {
  implicit def encoderIsF1[I, O](encoder: Encoder[I, O]): I => O =
    i => encoder.encode(i).right.get

  implicit def encoderIsF1WithError[I, O](encoder: Encoder[I, O]): I => OrError[O] =
    i => encoder.encode(i)
}